// Show loader for 2 seconds then transition to game
setTimeout(() => {
    const loader = document.getElementById('loader');
    const gameContainer = document.getElementById('gameContainer');
    const bgMusic = document.getElementById('bgMusic');
    
    loader.classList.add('fade-out');
    
    setTimeout(() => {
      gameContainer.classList.add('show');
      const playPromise = bgMusic.play();
      if (playPromise !== undefined) {
        playPromise.catch(error => {
          console.error("Auto-play was prevented", error);
          // We'll try to play again on the first user interaction
          document.body.addEventListener('click', () => bgMusic.play(), { once: true });
        });
      }
      startGameAnimation();
    }, 500);
  }, 2000);

  // Sidebar toggle for mobile (separate from user menu)
  const sidebarMenuIcon = document.getElementById('sidebarMenuIcon');
  const sidebar = document.getElementById('sidebar');
  const mainContent = document.getElementById('mainContent');

  if (sidebarMenuIcon) {
    sidebarMenuIcon.addEventListener('click', (event) => {
      event.stopPropagation();
      sidebar.classList.toggle('open');
    });
  }
  
  // Close sidebar if clicking outside of it on mobile
  document.addEventListener('click', function(event) {
      if (window.innerWidth <= 1024) {
          const isClickInsideSidebar = sidebar.contains(event.target);
          if (!isClickInsideSidebar && sidebar.classList.contains('open')) {
              sidebar.classList.remove('open');
          }
      }
  });

  // Randomly apply 'won' class to winning bet items
  document.addEventListener('DOMContentLoaded', function() {
      const winningBets = document.querySelectorAll('.bet-item.has-won');
      winningBets.forEach(bet => {
          // Give each winning bet a 50% chance of being highlighted
          if (Math.random() > 0.5) { 
              bet.classList.add('won');
          }
      });
  });

  function generateInitialBets() {
    const betsList = document.querySelector('.bets-list');
    if (!betsList) return;

    betsList.innerHTML = ''; // Clear existing bets

    const playerCount = Math.floor(Math.random() * 10) + 15; // 15 to 24 players

    for (let i = 0; i < playerCount; i++) {
        const randomUserId = `user${Math.random()}`;
        const playerName = `${Math.floor(Math.random() * 9) + 1}***${Math.floor(Math.random() * 9)}`;
        const betAmount = (Math.random() * 500 + 20).toFixed(2);

        const betItem = document.createElement('div');
        betItem.classList.add('bet-item');
        betItem.setAttribute('data-bet-amount', betAmount); // Store bet amount

        betItem.innerHTML = `
            <div class="bet-player">
              <div class="avatar" style="background-image: url('https://i.pravatar.cc/24?u=${randomUserId}')"></div>
              <span>${playerName}</span>
            </div>
            <div class="bet-amount">${betAmount}</div>
            <div></div>
            <div></div>
        `;
        
        betsList.appendChild(betItem);
    }
}

  // Weighted crash multiplier generator
  // Most crashes at low numbers (1.30, 2.35, etc.), rare above 7, very rare above 10
  function generateWeightedCrashMultiplier() {
      const rand = Math.random();
      let targetMultiplier;
      
      // 70% chance: Low range (1.02 - 3.50) - Most common crashes
      // Examples: 1.30, 1.85, 2.35, 2.90, 3.20, etc.
      if (rand < 0.70) {
          const lowMin = 1.02;
          const lowMax = 3.50;
          targetMultiplier = Math.random() * (lowMax - lowMin) + lowMin;
      }
      // 20% chance: Medium range (3.50 - 7.00) - Less common
      else if (rand < 0.90) {
          const mediumMin = 3.50;
          const mediumMax = 7.00;
          targetMultiplier = Math.random() * (mediumMax - mediumMin) + mediumMin;
      }
      // 8% chance: Higher range (7.00 - 10.00) - Rare
      else if (rand < 0.98) {
          const highMin = 7.00;
          const highMax = 10.00;
          targetMultiplier = Math.random() * (highMax - highMin) + highMin;
      }
      // 2% chance: Very high range (10.00 - 50.00) - Very rare, but possible
      else {
          const veryHighMin = 10.00;
          const veryHighMax = 50.00;
          targetMultiplier = Math.random() * (veryHighMax - veryHighMin) + veryHighMin;
      }
      
      // Round to 2 decimal places for clean display (like 1.30, 2.35, etc.)
      return Math.round(targetMultiplier * 100) / 100;
  }

  // Game Animation System
  function startGameAnimation() {
      const gameArea = document.querySelector('.game-area');
      const flyingPhase = document.getElementById('flyingPhase');
      const ufcPhase = document.getElementById('ufcPhase');
      const currentMultiplierEl = document.getElementById('currentMultiplier');
      const flewAwayText = document.getElementById('flewAwayText');
      
      const canvas = document.getElementById('gameCanvas');
      const ctx = canvas.getContext('2d');
      const plane = document.getElementById('plane');
      
      // Use SVG files from assets folder - loop through plane-0.svg to plane-3.svg
      // Get the base path from the initial plane image source (works with Laravel)
      const initialSrc = plane.src;
      const assetsBaseUrl = initialSrc.substring(0, initialSrc.lastIndexOf('/') + 1);
      const svgFiles = [
          assetsBaseUrl + 'plane-0.svg',
          assetsBaseUrl + 'plane-1.svg',
          assetsBaseUrl + 'plane-2.svg',
          assetsBaseUrl + 'plane-3.svg'
      ];
      let currentSvgIndex = 0;
      
      function animateSvgPlane() {
          // Loop through SVG files from 0 to 3 in a continuous cycle
          currentSvgIndex = (currentSvgIndex + 1) % svgFiles.length;
          plane.src = svgFiles[currentSvgIndex];
      }
      // Animate plane SVG every 200ms for smooth animation
      setInterval(animateSvgPlane, 200);

      let animationFrameId;

      function resizeCanvas() {
          // Account for padding when calculating canvas size
          const gameAreaStyle = window.getComputedStyle(gameArea);
          const paddingLeft = parseFloat(gameAreaStyle.paddingLeft) || 0;
          const paddingRight = parseFloat(gameAreaStyle.paddingRight) || 0;
          const paddingTop = parseFloat(gameAreaStyle.paddingTop) || 0;
          const paddingBottom = parseFloat(gameAreaStyle.paddingBottom) || 0;
          
          canvas.width = gameArea.clientWidth - paddingLeft - paddingRight;
          canvas.height = gameArea.clientHeight - paddingTop - paddingBottom;
      }
      window.addEventListener('resize', resizeCanvas);
      resizeCanvas();

      function easeInOutCubic(t) {
          return t < 0.5 ? 4 * t * t * t : 1 - Math.pow(-2 * t + 2, 3) / 2;
      }

      // Helper: Draw partial cubic Bezier from t=0 to t=t_plane
      function drawPartialBezier(ctx, p0, p1, p2, p3, t) {
          // De Casteljau's algorithm for cubic Bezier
          const steps = Math.max(10, Math.floor(t * 100));
          ctx.beginPath();
          ctx.moveTo(p0.x, p0.y);
          for (let i = 1; i <= steps; i++) {
              const tt = (i / steps) * t;
              const x =
                  Math.pow(1 - tt, 3) * p0.x +
                  3 * Math.pow(1 - tt, 2) * tt * p1.x +
                  3 * (1 - tt) * Math.pow(tt, 2) * p2.x +
                  Math.pow(tt, 3) * p3.x;
              const y =
                  Math.pow(1 - tt, 3) * p0.y +
                  3 * Math.pow(1 - tt, 2) * tt * p1.y +
                  3 * (1 - tt) * Math.pow(tt, 2) * p2.y +
                  Math.pow(tt, 3) * p3.y;
              ctx.lineTo(x, y);
          }
          ctx.strokeStyle = '#e60026';
          // Responsive line width: scales with canvas size
          const baseLineWidth = 3;
          const canvasScale = Math.min(canvas.width / 1920, canvas.height / 1080, 1);
          ctx.lineWidth = Math.max(2, baseLineWidth * Math.max(canvasScale, 0.5)); // Minimum 2px, scales up
          ctx.stroke();
          
          // Fill the area under the curve - fixed alignment
          const currentX = Math.pow(1 - t, 3) * p0.x + 3 * Math.pow(1 - t, 2) * t * p1.x + 3 * (1 - t) * Math.pow(t, 2) * p2.x + Math.pow(t, 3) * p3.x;
          ctx.lineTo(currentX, ctx.canvas.height);
          ctx.lineTo(p0.x, ctx.canvas.height);
          ctx.closePath();
          ctx.fillStyle = 'rgba(230, 0, 38, 0.25)';
          ctx.fill();
      }

      function runGameCycle() {
          if (animationFrameId) {
              cancelAnimationFrame(animationFrameId);
          }

          // Set game state to WAITING (betting allowed)
          onGameStateChange(GAME_STATES.WAITING, { nextRoundInMs: 3000 });

          // Reset states
          gameArea.classList.remove('flew-away-state');
          flyingPhase.classList.add('active');
          ufcPhase.classList.remove('active');
          currentMultiplierEl.classList.remove('is-finished');
          flewAwayText.classList.remove('show');
          plane.style.display = 'none';
          plane.style.transform = '';
          plane.classList.remove('is-waiting-next-round');
          currentMultiplierEl.textContent = '1.00x';

          generateInitialBets();

          // Wait for 3 seconds before starting the animation
          setTimeout(() => {
              // Generate unique round ID and random target multiplier
              const roundId = 'round_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
              const startTimestamp = Date.now();
              
              // Weighted random crash point - most crashes happen at low numbers
              // Lower numbers are common, higher numbers are rare (but still possible)
              // This makes the game challenging but winnable
              const targetMultiplier = generateWeightedCrashMultiplier();
              
              // Track round data
              currentRoundData = {
                  roundId: roundId,
                  startedAt: new Date(startTimestamp).toISOString(),
                  targetMultiplier: targetMultiplier,
                  userBetAmount: 0,
                  userWinAmount: 0,
                  userResult: 'no_bet'
              };
              
              // Check if user has active bets for this round
              let totalBetAmount = 0;
              Object.keys(betCardStates).forEach(betId => {
                  const betState = betCardStates[betId];
                  if (betState.state === 'pending' || betState.state === 'queued' || betState.state === 'cashout') {
                      totalBetAmount += betState.betAmount || 0;
                  }
              });
              
              if (totalBetAmount > 0) {
                  currentRoundData.userBetAmount = totalBetAmount;
                  currentRoundData.userResult = 'loss'; // Will be updated to 'win' if cashed out
              }

              // Set game state to PLAYING (betting closed, plane flying)
              onGameStateChange(GAME_STATES.PLAYING, { 
                  roundId: roundId,
                  startTimestamp: startTimestamp
              });

              let multiplier = 1.00;
              
              const planeTravelDuration = 7000; // Plane flies for 7000ms to reach end
              
              // Fixed constant increment - same increment amount throughout, never speeds up
              // This keeps the increment speed consistent regardless of target multiplier
              // For huge targets, it will take a long time (as intended)
              const constantIncrement = 0.01; // Always +0.01 per step (1.00 → 1.01 → 1.02 → 1.03...)
              // 500 BPM (4/4) = 500 beats/minute = 8.33 beats/second = 120ms per beat
              const timePerIncrement = 120; // 120ms per increment - 500 BPM speed for decimal increments
              
              // Calculate how many increments needed to reach target using constant increment
              // Formula: steps = (targetMultiplier - 1.00) / constantIncrement
              const totalSteps = Math.ceil((targetMultiplier - 1.00) / constantIncrement);
              
              // Flight duration is based on multiplier increments (consistent speed)
              // But plane will also reach end independently
              // Fly away happens when EITHER multiplier reaches target OR plane reaches end
              const multiplierFlightDuration = totalSteps * timePerIncrement;

              const startTime = Date.now();
              lastCashoutUpdateTime = 0; // Reset cashout update timer for new round
              const planeWidth = plane.offsetWidth || 0;
              const planeHeight = plane.offsetHeight || 0;
              const tailOffset = planeWidth * 0.1; // 10% from the left edge for tail position
              let p1, p2, p3;

              // Responsive path calculation - ensure plane stays visible
              // Canvas already accounts for padding, so use smaller margins
              const margin = Math.max(8, Math.min(canvas.width * 0.02, canvas.height * 0.02)); // Responsive margin inside canvas
              const actualPlaneWidth = plane.offsetWidth || 120;
              const actualPlaneHeight = plane.offsetHeight || 74;
              
              // Use different path for portrait mode (mobile)
              if (canvas.width < canvas.height) {
                // Mobile portrait - adjust path for proper stopping position
                p1 = { x: canvas.width * 0.5, y: canvas.height * 0.8 };
                p2 = { x: canvas.width * 0.7, y: canvas.height * 0.4 };
                // Final position should be at top-right area, accounting for plane size
                // Match desktop pattern but adjusted for portrait orientation
                p3 = { 
                  x: Math.min(canvas.width * 0.88, canvas.width - actualPlaneWidth - margin), 
                  y: Math.max(canvas.height * 0.15, actualPlaneHeight + margin)
                };
              } else { // Landscape for desktop
                p1 = { x: canvas.width * 0.4, y: canvas.height };
                p2 = { x: canvas.width * 0.7, y: canvas.height * 0.5 };
                // Ensure final position leaves room for plane
                p3 = { 
                  x: Math.min(canvas.width * 0.88, canvas.width - actualPlaneWidth - margin), 
                  y: Math.max(canvas.height * 0.15, actualPlaneHeight + margin)
                };
              }

              // Start position at the very bottom-left corner (clean, exact position as shown in image)
              // Path starts at bottom-left corner with small margin for clean spine appearance
              const p0 = { 
                x: margin, // Start with small margin from left edge
                y: canvas.height - margin // Start with small margin from bottom edge
              };

              // Track when plane first reaches idle position to smooth the transition
              let idleStartTime = null;
              const idleTransitionDuration = 500; // 500ms smooth transition into idle

              function animate(time) {
                  const elapsedTime = Date.now() - startTime;
                  
                  ctx.clearRect(0, 0, canvas.width, canvas.height);
                  
                  // --- Timing ---
                  // Plane and multiplier are independent
                  // Plane flies until it reaches the end (planeTravelDuration)
                  // Multiplier increments at fixed speed until it reaches target (multiplierFlightDuration)
                  // Fly away happens when EITHER condition is met
                  
                  let t_multiplier = Math.min(elapsedTime / multiplierFlightDuration, 1);
                  // t_multiplier represents progress through the multiplier steps (0 to 1)
                  
                  let t_plane = Math.min(elapsedTime / planeTravelDuration, 1);
                  
                  // Create a completely smooth, continuous easing without any pauses
                  // Use a single, smooth curve that starts very slow and gradually accelerates
                  t_plane = Math.pow(t_plane, 2.5); // Smooth power curve that eliminates any sudden changes
                  
                  // Check if we should fly away - either multiplier reached target OR plane reached end
                  const multiplierReachedTarget = t_multiplier >= 1;
                  const planeReachedEnd = t_plane >= 1;
                  const shouldFlyAway = multiplierReachedTarget || planeReachedEnd;

                  // --- Drawing ---
                  const cx = 3 * (p1.x - p0.x);
                  const bx = 3 * (p2.x - p1.x) - cx;
                  const ax = p3.x - p0.x - cx - bx;
                  const cy = 3 * (p1.y - p0.y);
                  const by = 3 * (p2.y - p1.y) - cy;
                  const ay = p3.y - p0.y - cy - by;

                  const xt = ax * t_plane*t_plane*t_plane + bx * t_plane*t_plane + cx * t_plane + p0.x;
                  const yt = ay * t_plane*t_plane*t_plane + by * t_plane*t_plane + cy * t_plane + p0.y;

                  let current_x = xt;
                  let current_y = yt;

                  // Apply idle animation offset to both plane and path end point
                  if (t_plane >= 1) {
                      // Track when idle state first starts
                      if (idleStartTime === null) {
                          idleStartTime = elapsedTime;
                      }
                      
                      // Calculate transition progress (0 to 1) for smooth entry into idle
                      const idleTransitionProgress = Math.min(
                          (elapsedTime - idleStartTime) / idleTransitionDuration,
                          1
                      );
                      
                      // Smooth ease-in function for transition
                      const smoothTransition = 1 - Math.pow(1 - idleTransitionProgress, 3);
                      
                      // Aviator-style dynamic idle movements
                      const time = elapsedTime / 1000; // Convert to seconds
                      
                      // Responsive idle movement - scales with canvas size
                      const movementScale = Math.min(canvas.width / 1920, canvas.height / 1080, 1.5);
                      const actualPlaneWidth = plane.offsetWidth || 120;
                      const actualPlaneHeight = plane.offsetHeight || 74;
                      
                      // More complex movement pattern with multiple frequencies
                      const idleOffsetX = Math.sin(time * 0.8) * 15 * movementScale + Math.sin(time * 1.2) * 8 * movementScale;
                      const idleOffsetY = Math.cos(time * 0.6) * 12 * movementScale + Math.sin(time * 1.5) * 6 * movementScale;
                      
                      // Add occasional larger movements (like Sportybet Aviator)
                      const randomMovement = Math.sin(time * 0.3) * 20 * movementScale;
                      
                      // Add dramatic downward drops (Sportybet style) - but clamp to keep in bounds
                      // Reduce dramatic drop on mobile - it's causing wrong stopping position
                      const isMobile = canvas.width < canvas.height;
                      const dropMultiplier = isMobile ? 20 : 80; // Much smaller drop on mobile
                      
                      const dramaticDrop = Math.sin(time * 0.15) * dropMultiplier * movementScale;
                      const dropIntensity = Math.max(0, Math.sin(time * 0.1) * 0.8 + 0.2);
                      
                      const finalOffsetX = idleOffsetX + randomMovement * 0.3;
                      const finalOffsetY = idleOffsetY + randomMovement * 0.4 + (dramaticDrop * dropIntensity);
                      
                      // Clamp offsets to ensure plane stays fully visible within canvas bounds
                      const maxOffsetX = Math.min(canvas.width - actualPlaneWidth - xt, xt - actualPlaneWidth * 0.5);
                      const maxOffsetY = Math.min(canvas.height - actualPlaneHeight - yt, yt - actualPlaneHeight * 0.5);
                      const clampedOffsetX = Math.max(-Math.abs(maxOffsetX), Math.min(Math.abs(maxOffsetX), finalOffsetX));
                      const clampedOffsetY = Math.max(-Math.abs(maxOffsetY), Math.min(Math.abs(maxOffsetY), finalOffsetY));
                      
                      // Apply smooth transition: start from zero offset, gradually fade in idle movement
                      const transitionedOffsetX = clampedOffsetX * smoothTransition;
                      const transitionedOffsetY = clampedOffsetY * smoothTransition;
                      
                      // Use the final bezier position as base for idle movements
                      current_x = xt + transitionedOffsetX;
                      current_y = yt + transitionedOffsetY;
                  } else {
                      // During flight, use the calculated bezier position
                      current_x = xt;
                      current_y = yt;
                      // Reset idle start time when not in idle
                      idleStartTime = null;
                  }

                  // Always use t_plane for the path progress
                  let t_for_path = Math.min(1, t_plane);

                  // Calculate elapsed time for idle animation - use same transition as plane
                  let px, py;
                  if (t_for_path >= 1) {
                      // Use the same bezier position as the plane for consistency
                      px = xt;
                      py = yt;
                      
                      // Apply same smooth transition for path consistency
                      const pathIdleStartTime = idleStartTime !== null ? idleStartTime : elapsedTime;
                      const pathIdleTransitionProgress = Math.min(
                          (elapsedTime - pathIdleStartTime) / idleTransitionDuration,
                          1
                      );
                      const pathSmoothTransition = 1 - Math.pow(1 - pathIdleTransitionProgress, 3);
                      
                      // Match the same Aviator-style movement pattern with clamping
                      const time = elapsedTime / 1000;
                      const movementScale = Math.min(canvas.width / 1920, canvas.height / 1080, 1.5);
                      const actualPlaneWidth = plane.offsetWidth || 120;
                      const actualPlaneHeight = plane.offsetHeight || 74;
                      
                      const idleOffsetX = Math.sin(time * 0.8) * 15 * movementScale + Math.sin(time * 1.2) * 8 * movementScale;
                      const idleOffsetY = Math.cos(time * 0.6) * 12 * movementScale + Math.sin(time * 1.5) * 6 * movementScale;
                      const randomMovement = Math.sin(time * 0.3) * 20 * movementScale;
                      
                      // Reduce dramatic drop on mobile - it's causing wrong stopping position
                      const isMobilePath = canvas.width < canvas.height;
                      const dropMultiplierPath = isMobilePath ? 20 : 80; // Much smaller drop on mobile
                      
                      const dramaticDrop = Math.sin(time * 0.15) * dropMultiplierPath * movementScale;
                      const dropIntensity = Math.max(0, Math.sin(time * 0.1) * 0.8 + 0.2);
                      
                      const finalOffsetX = idleOffsetX + randomMovement * 0.3;
                      const finalOffsetY = idleOffsetY + randomMovement * 0.4 + (dramaticDrop * dropIntensity);
                      
                      // Clamp for path as well
                      const maxOffsetX_path = Math.min(canvas.width - actualPlaneWidth - xt, xt - actualPlaneWidth * 0.5);
                      const maxOffsetY_path = Math.min(canvas.height - actualPlaneHeight - yt, yt - actualPlaneHeight * 0.5);
                      const clampedOffsetX_path = Math.max(-Math.abs(maxOffsetX_path), Math.min(Math.abs(maxOffsetX_path), finalOffsetX));
                      const clampedOffsetY_path = Math.max(-Math.abs(maxOffsetY_path), Math.min(Math.abs(maxOffsetY_path), finalOffsetY));
                      
                      // Apply smooth transition to path offsets as well
                      const transitionedPathOffsetX = clampedOffsetX_path * pathSmoothTransition;
                      const transitionedPathOffsetY = clampedOffsetY_path * pathSmoothTransition;
                      
                      // Add idle movement to the final flight position with smooth transition
                      px += transitionedPathOffsetX;
                      py += transitionedPathOffsetY;
                  } else {
                      // Calculate Bezier end point at t=t_for_path
                      px = Math.pow(1 - t_for_path, 3) * p0.x + 3 * Math.pow(1 - t_for_path, 2) * t_for_path * p1.x + 3 * (1 - t_for_path) * Math.pow(t_for_path, 2) * p2.x + Math.pow(t_for_path, 3) * p3.x;
                      py = Math.pow(1 - t_for_path, 3) * p0.y + 3 * Math.pow(1 - t_for_path, 2) * t_for_path * p1.y + 3 * (1 - t_for_path) * Math.pow(t_for_path, 2) * p2.y + Math.pow(t_for_path, 3) * p3.y;
                  }

                  // Draw the path up to the animated end point
                  if (t_for_path > 0.01) {
                      if (t_for_path >= 1) {
                          // During idle, animate the entire path with the plane
                          // Apply same smooth transition for path drawing
                          const pathDrawIdleStartTime = idleStartTime !== null ? idleStartTime : elapsedTime;
                          const pathDrawIdleTransitionProgress = Math.min(
                              (elapsedTime - pathDrawIdleStartTime) / idleTransitionDuration,
                              1
                          );
                          const pathDrawSmoothTransition = 1 - Math.pow(1 - pathDrawIdleTransitionProgress, 3);
                          
                          const time = elapsedTime / 1000;
                          const movementScale = Math.min(canvas.width / 1920, canvas.height / 1080, 1.5);
                          const actualPlaneWidth = plane.offsetWidth || 120;
                          const actualPlaneHeight = plane.offsetHeight || 74;
                          
                          const idleOffsetX = Math.sin(time * 0.8) * 15 * movementScale + Math.sin(time * 1.2) * 8 * movementScale;
                          const idleOffsetY = Math.cos(time * 0.6) * 12 * movementScale + Math.sin(time * 1.5) * 6 * movementScale;
                          const randomMovement = Math.sin(time * 0.3) * 20 * movementScale;
                          
                      // Reduce dramatic drop on mobile - it's causing wrong stopping position
                      const isMobileDraw = canvas.width < canvas.height;
                      const dropMultiplierDraw = isMobileDraw ? 20 : 80; // Much smaller drop on mobile
                      
                      const dramaticDrop = Math.sin(time * 0.15) * dropMultiplierDraw * movementScale;
                      const dropIntensity = Math.max(0, Math.sin(time * 0.1) * 0.8 + 0.2);
                      
                      const finalOffsetX = idleOffsetX + randomMovement * 0.3;
                      const finalOffsetY = idleOffsetY + randomMovement * 0.4 + (dramaticDrop * dropIntensity);
                      
                      // Clamp for path drawing as well
                          const maxOffsetX_draw = Math.min(canvas.width - actualPlaneWidth - xt, xt - actualPlaneWidth * 0.5);
                          const maxOffsetY_draw = Math.min(canvas.height - actualPlaneHeight - yt, yt - actualPlaneHeight * 0.5);
                          const clampedOffsetX_draw = Math.max(-Math.abs(maxOffsetX_draw), Math.min(Math.abs(maxOffsetX_draw), finalOffsetX));
                          const clampedOffsetY_draw = Math.max(-Math.abs(maxOffsetY_draw), Math.min(Math.abs(maxOffsetY_draw), finalOffsetY));
                          
                          // Apply smooth transition to path offsets
                          const transitionedPathDrawOffsetX = clampedOffsetX_draw * pathDrawSmoothTransition;
                          const transitionedPathDrawOffsetY = clampedOffsetY_draw * pathDrawSmoothTransition;
                          
                          // Apply idle offset to the end control point using same base as plane with transition
                          const animatedP3 = {
                              x: xt + transitionedPathDrawOffsetX,
                              y: yt + transitionedPathDrawOffsetY
                          };
                          
                          // Draw the complete path with the animated end point
                          drawPartialBezier(ctx, p0, p1, p2, animatedP3, 1);
                      } else {
                          // Normal path drawing during flight
                          drawPartialBezier(ctx, p0, p1, p2, p3, t_for_path);
                      }
                  } else if (t_plane >= 1) {
                      // Ensure path is drawn even during idle when t_for_path might be 0
                      // Apply same smooth transition
                      const pathFallbackIdleStartTime = idleStartTime !== null ? idleStartTime : elapsedTime;
                      const pathFallbackIdleTransitionProgress = Math.min(
                          (elapsedTime - pathFallbackIdleStartTime) / idleTransitionDuration,
                          1
                      );
                      const pathFallbackSmoothTransition = 1 - Math.pow(1 - pathFallbackIdleTransitionProgress, 3);
                      
                      const time = elapsedTime / 1000;
                      const movementScale = Math.min(canvas.width / 1920, canvas.height / 1080, 1.5);
                      const actualPlaneWidth = plane.offsetWidth || 120;
                      const actualPlaneHeight = plane.offsetHeight || 74;
                      
                      const idleOffsetX = Math.sin(time * 0.8) * 15 * movementScale + Math.sin(time * 1.2) * 8 * movementScale;
                      const idleOffsetY = Math.cos(time * 0.6) * 12 * movementScale + Math.sin(time * 1.5) * 6 * movementScale;
                      const randomMovement = Math.sin(time * 0.3) * 20 * movementScale;
                      
                      // Reduce dramatic drop on mobile - it's causing wrong stopping position
                      const isMobileFallback = canvas.width < canvas.height;
                      const dropMultiplierFallback = isMobileFallback ? 20 : 80; // Much smaller drop on mobile
                      
                      const dramaticDrop = Math.sin(time * 0.15) * dropMultiplierFallback * movementScale;
                      const dropIntensity = Math.max(0, Math.sin(time * 0.1) * 0.8 + 0.2);
                      
                      const finalOffsetX = idleOffsetX + randomMovement * 0.3;
                      const finalOffsetY = idleOffsetY + randomMovement * 0.4 + (dramaticDrop * dropIntensity);
                      
                      // Clamp for fallback path as well
                      const maxOffsetX_fallback = Math.min(canvas.width - actualPlaneWidth - xt, xt - actualPlaneWidth * 0.5);
                      const maxOffsetY_fallback = Math.min(canvas.height - actualPlaneHeight - yt, yt - actualPlaneHeight * 0.5);
                      const clampedOffsetX_fallback = Math.max(-Math.abs(maxOffsetX_fallback), Math.min(Math.abs(maxOffsetX_fallback), finalOffsetX));
                      const clampedOffsetY_fallback = Math.max(-Math.abs(maxOffsetY_fallback), Math.min(Math.abs(maxOffsetY_fallback), finalOffsetY));
                      
                      // Apply smooth transition
                      const transitionedPathFallbackOffsetX = clampedOffsetX_fallback * pathFallbackSmoothTransition;
                      const transitionedPathFallbackOffsetY = clampedOffsetY_fallback * pathFallbackSmoothTransition;
                      
                      const animatedP3 = {
                          x: xt + transitionedPathFallbackOffsetX,
                          y: yt + transitionedPathFallbackOffsetY
                      };
                      
                      drawPartialBezier(ctx, p0, p1, p2, animatedP3, 1);
                  }

                  // --- Plane ---
                  plane.style.display = 'block';
                  plane.style.transform = `translate(-10%, -50%)`; // Position tail at path point
                  
                  // Get canvas position relative to game area to account for padding (no left padding)
                  const canvasRect = canvas.getBoundingClientRect();
                  const gameAreaRect = gameArea.getBoundingClientRect();
                  const canvasOffsetX = canvasRect.left - gameAreaRect.left; // Will be 0 (no left padding)
                  const canvasOffsetY = canvasRect.top - gameAreaRect.top;
                  
                  // Position plane accounting for canvas offset (no left padding)
                  // Transform is translate(-10%, -50%) so center of plane is at path point
                  const planeHeight = plane.offsetHeight || 0;
                  plane.style.left = `${canvasOffsetX + px}px`;
                  plane.style.top = `${canvasOffsetY + py - planeHeight / 2}px`; // Center plane at path point (transform accounts for -50%)

                  // --- Multiplier ---
                  // Constant increment: 1.00 → 1.02 → 1.04 → 1.06 → 1.08, etc.
                  // Each step adds the same amount: always +0.02 per step
                  // Increment speed is consistent (fixed timePerIncrement) regardless of target
                  // For huge targets, it takes a long time (as intended)
                  const baseMultiplier = 1.00;
                  
                  // Calculate current multiplier using constant increment
                  const currentStep = Math.floor(t_multiplier * totalSteps);
                  
                  // Calculate multiplier at current step: baseMultiplier + (step * constantIncrement)
                  let currentMultiplier = baseMultiplier + (currentStep * constantIncrement);
                  
                  // Clamp current multiplier to not exceed target
                  if (currentMultiplier > targetMultiplier) {
                      currentMultiplier = targetMultiplier;
                  }
                  
                  // Calculate the next step multiplier (for interpolation)
                  let nextStepMultiplier = baseMultiplier + ((currentStep + 1) * constantIncrement);
                  if (nextStepMultiplier > targetMultiplier) {
                      nextStepMultiplier = targetMultiplier;
                  }
                  
                  // Interpolate between steps for smooth display (constant increment throughout)
                  const stepProgress = (t_multiplier * totalSteps) - currentStep;
                  multiplier = currentMultiplier + (nextStepMultiplier - currentMultiplier) * stepProgress;
                  
                  // Clamp to target
                  if (multiplier >= targetMultiplier) {
                      multiplier = targetMultiplier;
                  }
                  
                  // Always update to show smooth increments
                  currentMultiplierEl.textContent = multiplier.toFixed(2) + 'x';

                  // Update bet card cashout values at 195 BPM (4/4) for button display
                  // 195 BPM = 195 beats/minute = 3.25 beats/second = ~308ms per update
                  updateBetCardCashoutValuesAt195BPM(multiplier, elapsedTime);

                  // --- Real-time Bet Wins ---
                  document.querySelectorAll('.bet-item:not(.won)').forEach(betItem => {
                    // Lower probability on each frame to make wins less frequent
                    if (Math.random() < 1.002) {
                        const betAmount = parseFloat(betItem.getAttribute('data-bet-amount'));
                        const winAmount = (betAmount * multiplier).toFixed(2);
                        const colors = ['blue', 'green', 'purple', 'red'];
                        const randomColor = colors[Math.floor(Math.random() * colors.length)];

                        betItem.classList.add('won');
                        betItem.children[2].innerHTML = `<div class="bet-multiplier ${randomColor}">${multiplier.toFixed(2)}x</div>`;
                        betItem.children[3].innerHTML = `<div class="bet-win">${winAmount}</div>`;
                    }
                  });
                  
                  // --- State Logic ---
                  // Fly away happens when multiplier reaches target
                  // If plane reaches end first, plane idles while multiplier continues incrementing
                  // If multiplier reaches target first, instant fly away regardless of plane position
                  
                  if (!multiplierReachedTarget) {
                      // Continue animation - multiplier still incrementing
                      // If plane reached end, it will idle (handled in drawing section above)
                      // If plane hasn't reached end, it continues flying
                      animationFrameId = requestAnimationFrame(animate);
                  } else {
                      // Multiplier reached target - instant fly away
                      // One of the conditions is met - trigger fly away
                      let crashMultiplier;
                      if (multiplierReachedTarget) {
                          // Multiplier reached target - instant fly away at current plane position
                          crashMultiplier = targetMultiplier;
                      } else if (planeReachedEnd && !multiplierReachedTarget) {
                          // Plane reached end, but multiplier hasn't - use current multiplier as crash point
                          crashMultiplier = multiplier;
                      } else {
                          crashMultiplier = multiplier; // Fallback
                      }
                      
                      // FLEW AWAY STATE - HAPPENS INSTANTLY
                      // Play sprite audio with proper timing
                      const spriteAudio = document.getElementById('spriteAudio');
                      if (spriteAudio && spriteAudio.src) {
                        spriteAudio.currentTime = 0;
                        spriteAudio.play().catch(e => {
                          console.log('Sprite audio play failed:', e);
                        });
                      }
                      
                      gameArea.classList.add('flew-away-state');
                      currentMultiplierEl.classList.add('is-finished');
                      flewAwayText.classList.add('show');
                      
                      // Store round and multiplier in database
                      storeGameRound(crashMultiplier);
                      
                      // Update round result for uncashed bets
                      if (currentRoundData.userBetAmount > 0 && currentRoundData.userWinAmount === 0) {
                          currentRoundData.userResult = 'loss';
                      }
                      
                      // Store game round and multiplier in database
                      storeGameRound(crashMultiplier);
                      
                      // Set game state to ENDED with the crash multiplier
                      onGameStateChange(GAME_STATES.ENDED, { finalMultiplier: crashMultiplier });
                      
                      // Set bet cards to Cancel/Waiting state
                      setBetCardStatesToWaiting();
                      
                      // Use the actual current position for flew away animation
                      const startX = px;
                      const startY = py;
                      
                      const flewAwayStartTime = Date.now();
                      function animateFlewAway() {
                          const flewAwayElapsed = Date.now() - flewAwayStartTime;
                          const progress = flewAwayElapsed / 200; // Much faster flew away animation
                          
                          // Fly away diagonally upward and to the right
                          const newX = startX + progress * (canvas.width * 1.2);
                          const newY = startY - progress * (canvas.height * 0.3); // Fly upward
                          
                          if (progress < 1) {
                              plane.style.left = `${newX}px`;
                              plane.style.top = `${newY}px`;
                              
                              // Fade out the bezier line along with the plane
                              const fadeAlpha = 1 - progress; // Fade from 1 to 0
                              ctx.clearRect(0, 0, canvas.width, canvas.height);
                              
                              // Redraw the bezier line with fading opacity
                              ctx.globalAlpha = fadeAlpha;
                              drawPartialBezier(ctx, p0, p1, p2, p3, 1);
                              ctx.globalAlpha = 1; // Reset alpha
                              
                              requestAnimationFrame(animateFlewAway);
                          } else {
                              // Hide plane after flew away animation
                              plane.style.display = 'none';
                              ctx.clearRect(0, 0, canvas.width, canvas.height);
                          }
                      }
                      animateFlewAway();

                      // Transition to UFC phase after a delay
                      setTimeout(() => {
                          flyingPhase.classList.remove('active');
                          ufcPhase.classList.add('active');
                          
                          // Hide flew away text during UFC phase
                          flewAwayText.classList.remove('show');
                          
                          // Position plane at bottom-left corner when UFC phase starts
                          // Account for padding when positioning
                          const canvasRect = canvas.getBoundingClientRect();
                          const gameAreaRect = gameArea.getBoundingClientRect();
                          const canvasOffsetX = canvasRect.left - gameAreaRect.left;
                          const canvasOffsetY = canvasRect.top - gameAreaRect.top;
                          
                          // Position at bottom-left corner (as shown in image)
                          const planeHeightWaiting = plane.offsetHeight || 74;
                          plane.style.left = `${canvasOffsetX + 10}px`; // Small offset from left edge
                          plane.style.right = 'auto';
                          plane.style.bottom = `${gameAreaRect.bottom - canvasRect.bottom + 10}px`; // Small offset from bottom
                          plane.style.top = 'auto';
                          plane.style.transform = 'none'; // Remove transform to position at bottom-left
                          plane.classList.add('is-waiting-next-round');
                          plane.style.display = 'block';
                          
                          const progressBar = document.querySelector('.ufc-progress-bar');
                          if (progressBar) {
                              progressBar.style.width = '100%';
                              const ufcStartTime = Date.now();
                              const duration = 5000;
                              const animateProgress = () => {
                                  const elapsed = Date.now() - ufcStartTime;
                                  const progress = Math.max(0, 100 - (elapsed / duration) * 100);
                                  progressBar.style.width = progress + '%';
                                  if (elapsed < duration) requestAnimationFrame(animateProgress);
                              };
                              animateProgress();
                          }
                          
                          setTimeout(() => {
                              ufcPhase.classList.remove('active');
                              runGameCycle();
                          }, 5000);
                      }, 2000);
                  }
              }
              animationFrameId = requestAnimationFrame(animate);
          }, 3000);
      }
      
      runGameCycle();
  }

  // Chat functionality
  document.addEventListener('DOMContentLoaded', function() {
      // Chat close button
      const chatClose = document.querySelector('.chat-close');
      const chatSidebar = document.querySelector('.chat-sidebar');
      
      if (chatClose && chatSidebar) {
          chatClose.addEventListener('click', () => {
              chatSidebar.style.display = 'none';
          });
      }
      
      // Chat like buttons
      const likeButtons = document.querySelectorAll('.chat-like-btn');
      likeButtons.forEach(button => {
          button.addEventListener('click', function() {
              const countElement = this.parentNode.querySelector('.chat-like-count');
              let count = parseInt(countElement.textContent);
              count++;
              countElement.textContent = count;
              
              // Visual feedback
              this.style.color = '#4ade80';
              setTimeout(() => {
                  this.style.color = '';
              }, 200);
          });
      });
      
      // Auto-scroll chat to bottom
      const chatMessages = document.querySelector('.chat-messages');
      if (chatMessages) {
          chatMessages.scrollTop = chatMessages.scrollHeight;
      }
  });

  // --- BET CARD STATE MANAGEMENT ---
  let betCardStates = {
      bet1: { state: 'bet', betAmount: 2.00, cashoutValue: 0, cashedOut: false, isActive: false },
      bet2: { state: 'bet', betAmount: 2.00, cashoutValue: 0, cashedOut: false, isActive: false }
  };

  // Auto bet and auto cashout settings per panel
  let autoSettings = {
      bet1: { autoBetEnabled: false, autoCashoutEnabled: false, autoCashoutMultiplier: 1.10 },
      bet2: { autoBetEnabled: false, autoCashoutEnabled: false, autoCashoutMultiplier: 1.10 }
  };

  // Store original bet amounts to preserve them during resets
  let originalBetAmounts = {
      bet1: 2.00,
      bet2: 2.00
  };

  // Track current multiplier for real-time updates
  let currentMultiplier = 1.00;

  // Define helper functions early so they're available for onclick handlers
  // Helper function for quick bet buttons
  function setQuickBet(amount, panelIndex) {
    const betId = panelIndex === 0 ? 'bet1' : 'bet2';
    const betState = betCardStates[betId];
    
    // Don't allow changing bet amount if bet is already active
    if (betState && (betState.state === 'pending' || betState.state === 'cashout' || betState.state === 'queued')) {
      return;
    }
    
    betCardStates[betId].betAmount = amount;
    originalBetAmounts[betId] = amount;
    syncBetAmountDisplay(panelIndex);
  }
  
  // Helper function for change bet buttons
  function changeBet(delta, panelIndex) {
    const betId = panelIndex === 0 ? 'bet1' : 'bet2';
    const betState = betCardStates[betId];
    
    // Don't allow changing bet amount if bet is already active
    if (betState && (betState.state === 'pending' || betState.state === 'cashout' || betState.state === 'queued')) {
      return;
    }
    
    let currentAmount = betState.betAmount;
    const newAmount = Math.max(1, Math.min(currentAmount + delta, 1000000));
    betCardStates[betId].betAmount = newAmount;
    originalBetAmounts[betId] = newAmount;
    syncBetAmountDisplay(panelIndex);
  }
  
  // Sync bet amount display
  function syncBetAmountDisplay(panelIndex) {
    const betId = panelIndex === 0 ? 'bet1' : 'bet2';
    const betAmount = betCardStates[betId].betAmount;
    
    // Update bet amount in both bet-content and auto-content
    const panel = document.querySelectorAll('.betting-panel')[panelIndex];
    if (panel) {
      const betAmountElements = panel.querySelectorAll('.bet-amount');
      betAmountElements.forEach(el => {
        el.textContent = betAmount.toFixed(2);
      });
      
      const betBtnAmountElements = panel.querySelectorAll('.bet-btn-amount');
      betBtnAmountElements.forEach(el => {
        if (betCardStates[betId].state === 'bet') {
          el.textContent = betAmount.toFixed(2) + ' ZAR';
        }
      });
    }
  }
  
  // Make functions global immediately
  window.setQuickBet = setQuickBet;
  window.changeBet = changeBet;

  // Game round tracking
  let currentRoundData = {
      roundId: null,
      startedAt: null,
      targetMultiplier: null,
      userBetAmount: 0,
      userWinAmount: 0,
      userResult: 'no_bet'
  };
  
  // Load flew away multipliers from database
  async function loadFlewAwayMultipliers() {
      try {
          const response = await fetch('/api/flew-away-multipliers?limit=50', {
              headers: {
                  'X-Requested-With': 'XMLHttpRequest',
                  'Accept': 'application/json',
              },
              credentials: 'same-origin',
          });
          
          if (response.ok) {
              const data = await response.json();
              if (data.success && data.multipliers && data.multipliers.length > 0) {
                  // Clear existing multipliers display
                  const multipliersDiv = document.querySelector('.multipliers');
                  if (multipliersDiv) {
                      multipliersDiv.innerHTML = '';
                      
                      // Add multipliers from database (newest first)
                      data.multipliers.forEach(item => {
                          const span = document.createElement('span');
                          const colors = ['blue', 'purple', 'red'];
                          const randomColor = colors[Math.floor(Math.random() * colors.length)];
                          span.className = `multiplier ${randomColor}`;
                          span.textContent = parseFloat(item.multiplier).toFixed(2) + 'x';
                          multipliersDiv.insertBefore(span, multipliersDiv.firstChild);
                      });
                      
                      // Limit to 40 multipliers displayed
                      while (multipliersDiv.children.length > 40) {
                          multipliersDiv.removeChild(multipliersDiv.lastChild);
                      }
                  }
                  console.log('Loaded', data.multipliers.length, 'flew away multipliers from database');
              } else {
                  // No multipliers in database yet, keep default ones
                  console.log('No multipliers found in database, using default display');
              }
          } else {
              console.warn('Failed to load flew away multipliers:', response.status);
          }
      } catch (error) {
          console.error('Failed to load flew away multipliers:', error);
      }
      
      // Always return a resolved promise so initialization can continue
      return Promise.resolve();
  }
  
  // Store game round and multiplier in database
  async function storeGameRound(finalMultiplier) {
      try {
          if (!currentRoundData.roundId) {
              console.warn('Cannot store game round: no round ID');
              return;
          }
          
          const endedAt = new Date().toISOString();
          const payload = {
              round_id: currentRoundData.roundId,
              multiplier: finalMultiplier,
              bet_amount: currentRoundData.userBetAmount,
              win_amount: currentRoundData.userWinAmount,
              result: currentRoundData.userResult,
              started_at: currentRoundData.startedAt,
              ended_at: endedAt
          };
          
          const response = await fetch('/api/game-round', {
              method: 'POST',
              headers: {
                  'Content-Type': 'application/json',
                  'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || '',
                  'Accept': 'application/json',
                  'X-Requested-With': 'XMLHttpRequest',
              },
              credentials: 'same-origin',
              body: JSON.stringify(payload)
          });
          
          if (response.ok) {
              const data = await response.json();
              if (data.success) {
                  console.log('Game round stored successfully:', currentRoundData.roundId);
                  // Add to display immediately and refresh from database
                  addFlewAwayMultiplier(finalMultiplier);
                  // Also refresh from database to get latest
                  setTimeout(() => {
                      loadFlewAwayMultipliers();
                  }, 500);
              }
          } else {
              console.error('Failed to store game round:', response.status);
          }
      } catch (error) {
          console.error('Error storing game round:', error);
      }
  }

  function resetBetCardStates() {
      Object.keys(betCardStates).forEach(betId => {
          // Only reset if not queued
          if (betCardStates[betId].state !== 'queued') {
              const panelIndex = betId === 'bet1' ? 0 : 1;
              // Use the stored original amount
              const currentAmount = originalBetAmounts[betId];
              
              betCardStates[betId] = {
                  state: 'bet',
                  betAmount: currentAmount,
                  cashoutValue: 0,
                  cashedOut: false,
                  isActive: false
              };
              updateBetButtonState(panelIndex, 'bet');
          } else {
              // If queued, keep the state and update the button to queued
              const panelIndex = betId === 'bet1' ? 0 : 1;
              updateBetButtonState(panelIndex, 'queued');
          }
      });
  }
  function setBetCardStatesToWaiting() {
      Object.keys(betCardStates).forEach(betId => {
          const betState = betCardStates[betId];
          const panelIndex = betId === 'bet1' ? 0 : 1;
          
          // If user had an active bet that wasn't cashed out, mark as lost
          if (betState.state === 'cashout' && !betState.cashedOut && betState.isActive) {
              // Mark as lost
              betCardStates[betId].state = 'lost';
              updateBetButtonState(panelIndex, 'lost');
              
              // Reset to original bet amount after showing lost state
              setTimeout(() => {
                  const originalAmount = originalBetAmounts[betId];
                  
                  betCardStates[betId] = {
                      state: 'bet',
                      betAmount: originalAmount,
                      cashoutValue: 0,
                      cashedOut: false,
                      isActive: false
                  };
                  updateBetButtonState(panelIndex, 'bet');
                  updateBetControlsState(panelIndex, false);
                  
                  // If auto bet is enabled, place bet automatically
                  if (autoSettings[betId].autoBetEnabled && originalAmount <= userBalance) {
                      if (placeBet(originalAmount, panelIndex)) {
                          updateBetButtonState(panelIndex, 'pending');
                          updateBetControlsState(panelIndex, true);
                      }
                  }
              }, 2000); // Show lost state for 2 seconds, then reset
          } else if (betState.state === 'queued') {
              // Keep queued bets as is - they'll be activated in next round
              // Do nothing, keep the queued state
              updateBetControlsState(panelIndex, true);
          } else if (betState.state !== 'cashedout' && betState.state !== 'lost') {
              // Reset other states to bet state
              const originalAmount = originalBetAmounts[betId];
              betCardStates[betId] = {
                  state: 'bet',
                  betAmount: originalAmount,
                  cashoutValue: 0,
                  cashedOut: false,
                  isActive: false
              };
              updateBetButtonState(panelIndex, 'bet');
              updateBetControlsState(panelIndex, false);
              
              // If auto bet is enabled, place bet automatically
              if (autoSettings[betId].autoBetEnabled && originalAmount <= userBalance) {
                  if (placeBet(originalAmount, panelIndex)) {
                      updateBetButtonState(panelIndex, 'pending');
                      updateBetControlsState(panelIndex, true);
                  }
              }
          } else if (betState.state === 'cashedout') {
              // Reset cashed out state after delay (handled in cashOut function)
              updateBetControlsState(panelIndex, false);
          }
      });
  }

  // Track last update time for 195 BPM throttling
  let lastCashoutUpdateTime = 0;
  const cashoutUpdateInterval = 1000 / (195 / 60); // 195 BPM = ~308ms per update

  function updateBetCardCashoutValuesAt195BPM(multiplier, elapsedTime) {
      currentMultiplier = multiplier;
      
      // Throttle updates to 195 BPM (4/4) - update every ~308ms
      if (elapsedTime - lastCashoutUpdateTime >= cashoutUpdateInterval) {
          lastCashoutUpdateTime = elapsedTime;
          
          Object.keys(betCardStates).forEach(betId => {
              const betState = betCardStates[betId];
              if (betState.state === 'cashout' && betState.isActive) {
                  // Calculate winning amount: betAmount × multiplier (doubled by bet amount)
                  betState.cashoutValue = betState.betAmount * multiplier;
                  const panelIndex = betId === 'bet1' ? 0 : 1;
                  updateBetButtonState(panelIndex, 'cashout');
                  
                  // Auto cashout: If auto cashout is enabled and multiplier reached target
                  if (autoSettings[betId].autoCashoutEnabled) {
                      const targetMultiplier = autoSettings[betId].autoCashoutMultiplier;
                      if (multiplier >= targetMultiplier) {
                          // Auto cashout this bet
                          cashOut(panelIndex);
                      }
                  }
              }
          });
      }
  }

  function updateBetCardCashoutValues(multiplier) {
      currentMultiplier = multiplier;
      Object.keys(betCardStates).forEach(betId => {
          if (betCardStates[betId].state === 'cashout' && betCardStates[betId].isActive) {
              // Calculate winning amount: betAmount × multiplier (doubled by bet amount)
              betCardStates[betId].cashoutValue = betCardStates[betId].betAmount * multiplier;
              updateBetButtonState(betId === 'bet1' ? 0 : 1, 'cashout');
          }
      });
  }

  // Update bet button state
  function updateBetButtonState(panelIndex, state) {
    const panel = document.querySelectorAll('.betting-panel')[panelIndex];
    if (!panel) return;
    
    // Update all bet buttons in this panel (both bet-content and auto-content)
    const betBtns = panel.querySelectorAll('.main-bet-btn');
    const betId = panelIndex === 0 ? 'bet1' : 'bet2';
    const betState = betCardStates[betId];
    
    betBtns.forEach(betBtn => {
      const betBtnText = betBtn.querySelector('.bet-btn-text');
      const betBtnAmount = betBtn.querySelector('.bet-btn-amount');
      if (!betBtnText || !betBtnAmount) return;
      
      betBtn.className = 'main-bet-btn';
      switch (state) {
        case 'pending':
          betBtnText.textContent = 'Cancel';
          betBtnAmount.textContent = `${betState.betAmount.toFixed(2)} ZAR`;
          betBtn.classList.add('pending');
          betBtnAmount.style.fontSize = '';
          betBtnAmount.style.lineHeight = '';
          break;
        case 'queued':
          betBtnText.textContent = 'Cancel';
          betBtnAmount.textContent = 'Waiting for next round';
          betBtn.classList.add('queued');
          betBtnAmount.style.fontSize = '13px';
          betBtnAmount.style.lineHeight = '1.3';
          break;
        case 'cashout':
          betBtnText.textContent = 'Cash Out';
          betBtnAmount.textContent = `${betState.cashoutValue.toFixed(2)} ZAR`;
          betBtn.classList.add('cashout');
          betBtnAmount.style.fontSize = '';
          betBtnAmount.style.lineHeight = '';
          break;
        case 'cashedout':
          betBtnText.textContent = 'Cashed Out!';
          betBtnAmount.textContent = `${betState.cashoutValue.toFixed(2)} ZAR`;
          betBtn.classList.add('cashedout');
          break;
        case 'lost':
          betBtnText.textContent = 'Lost';
          betBtnAmount.textContent = '-';
          betBtn.classList.add('lost');
          break;
        default:
          betBtnText.textContent = 'Bet';
          betBtnAmount.textContent = `${betState.betAmount.toFixed(2)} ZAR`;
          betBtnAmount.style.fontSize = '';
          betBtnAmount.style.lineHeight = '';
          break;
      }
    });
  }

  // --- USER BALANCE MANAGEMENT ---
  let userBalance = 0.00;
  window.userBalance = 0.00; // Make accessible globally
  let gameUuid = null;

  // Initialize balance from server
  async function initializeBalance() {
      try {
          const response = await fetch('/api/balance', {
              headers: {
                  'X-Requested-With': 'XMLHttpRequest',
                  'Accept': 'application/json',
              },
              credentials: 'same-origin',
          });
          
          if (response.ok) {
              const data = await response.json();
              userBalance = parseFloat(data.balance || 0);
              window.userBalance = userBalance; // Update global
              updateBalanceDisplay();
          } else {
              // Handle error response
              const errorData = await response.json().catch(() => ({}));
              console.warn('Balance API returned error:', response.status, errorData);
              // Set default balance on error
              userBalance = 0;
              window.userBalance = 0;
              updateBalanceDisplay();
          }
      } catch (error) {
          console.error('Failed to load balance:', error);
          // Set default balance on error
          userBalance = 0;
          window.userBalance = 0;
          updateBalanceDisplay();
      }
  }
  window.initializeBalance = initializeBalance; // Make accessible globally

  function updateBalanceDisplay() {
      const balanceEl = document.getElementById('userBalance');
      if (balanceEl) {
          balanceEl.textContent = userBalance.toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ',');
      }
  }
  window.updateBalanceDisplay = updateBalanceDisplay; // Make accessible globally

  // Initialize on page load
  initializeBalance();
  
  // Generate game UUID for this session
  gameUuid = 'game_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);

  // --- GAME STATE MANAGEMENT SYSTEM ---
  const GAME_STATES = {
      WAITING: 'waiting',
      PLAYING: 'playing', 
      ENDED: 'ended'
  };

  let currentGameState = GAME_STATES.WAITING;
  let userHasActiveBet = false;
  let currentRoundId = null;
  let countdownInterval = null;
  let autoBetEnabled = false;
  let lastBetAmount = 10.00;
  let finalMultiplier = 0;

  // UI Elements
  const betButton = document.getElementById('bet1-action');
  const maxBetButton = document.querySelector('.max-bet-button') || document.createElement('button');
  const autoBetToggle = document.querySelector('.auto-bet-toggle') || document.createElement('button');
  const cashOutButton = document.querySelector('.cash-out-button') || betButton; // Use bet button for now
  const countdownDisplay = document.querySelector('.countdown-display') || document.createElement('div');

  // --- GAME STATE TRANSITIONS ---
  function onGameStateChange(newState, payload = {}) {
      console.log(`Game state changed from ${currentGameState} to ${newState}`, payload);
      currentGameState = newState;
      
      switch (newState) {
          case GAME_STATES.WAITING:
              handleWaitingState(payload);
              break;
          case GAME_STATES.PLAYING:
              handlePlayingState(payload);
              break;
          case GAME_STATES.ENDED:
              handleEndedState(payload);
              break;
      }
      
      updateButtons();
  }

  function handleWaitingState(payload) {
    Object.keys(betCardStates).forEach(betId => {
          const betState = betCardStates[betId];
          const panelIndex = betId === 'bet1' ? 0 : 1;
          if (betState.state !== 'queued') {
              // Reset all other bets to default using stored original amount
              const currentAmount = originalBetAmounts[betId];
              
              betCardStates[betId] = {
                  state: 'bet',
                  betAmount: currentAmount,
                  cashoutValue: 0,
                  cashedOut: false,
                  isActive: false
              };
              updateBetButtonState(panelIndex, 'bet');
              updateBetControlsState(panelIndex, false);
              
              // Auto bet: If auto bet is enabled, automatically place bet
              if (autoSettings[betId].autoBetEnabled && currentAmount <= userBalance) {
                  if (placeBet(currentAmount, panelIndex)) {
                      updateBetButtonState(panelIndex, 'pending');
                      updateBetControlsState(panelIndex, true);
                  }
              }
          } else {
              updateBetButtonState(panelIndex, 'queued');
              updateBetControlsState(panelIndex, true);
          }
      });
      // Start countdown if provided
      if (payload.nextRoundInMs) {
          startCountdown(payload.nextRoundInMs);
      }
      // Update UI
      if (countdownDisplay) {
          countdownDisplay.style.display = 'block';
      }
      
      // Start dynamic betting stats updates during waiting phase
      startDynamicBettingStats();
  }

  function handlePlayingState(payload) {
      stopCountdown();
      currentRoundId = payload.roundId;
      Object.keys(betCardStates).forEach(betId => {
          const betState = betCardStates[betId];
          if (betState.state === 'pending' || betState.state === 'queued') {
              betCardStates[betId] = {
                  ...betState,
                  state: 'cashout',
                  isActive: true
              };
              const panelIndex = betId === 'bet1' ? 0 : 1;
              updateBetButtonState(panelIndex, 'cashout');
          }
      });
      // Update UI
      if (countdownDisplay) {
          countdownDisplay.style.display = 'none';
      }
      
      // Continue dynamic betting stats during playing phase (until plane flies away)
      // Don't stop the stats - let them continue to increase
      // The stats will only stop when the game ends (plane flies away)
      
      // Increase update rate during playing phase for more intensity
      if (bettingStatsInterval) {
        clearInterval(bettingStatsInterval);
        bettingStatsInterval = setInterval(() => {
          // More aggressive updates during flight
          const newBets = Math.floor(Math.random() * 8) + 2; // 2-9 new bets
          targetBettingStats.currentBets += newBets;
          
          // More frequent total bet increases
          if (Math.random() < 0.6) {
            targetBettingStats.totalBets += Math.floor(Math.random() * 4) + 1;
          }
          
          // Bigger win amount increases
          targetBettingStats.totalWin += (Math.random() * 150 + 30);
          
          // More dramatic player count changes during flight (but still controlled)
          if (Math.random() < 0.7) { // 70% chance to update player count
            const playerChange = Math.floor(Math.random() * 25) - 8; // -8 to +17
            targetPlayersCount = Math.max(180, Math.min(600, targetPlayersCount + playerChange));
            animatePlayersCount();
          }
          
          animateBettingStats();
          addNewBetItems(newBets);
          
        }, Math.random() * 1200 + 600); // Moderate updates: 0.6-1.8 seconds
      }
  }

  function handleEndedState(payload) {
      finalMultiplier = payload.finalMultiplier || 0;
      addFlewAwayMultiplier(finalMultiplier);
      
      // Show results
      showGameResults(payload);
      
      // Mark uncashed bets as lost
      setBetCardStatesToWaiting();
      
      // Stop dynamic betting stats updates during ended phase
      stopDynamicBettingStats();
      
      // Transition back to waiting after 5 seconds
      setTimeout(() => {
          onGameStateChange(GAME_STATES.WAITING, { nextRoundInMs: 5000 });
      }, 5000);
  }

  // --- BUTTON MANAGEMENT ---
  function updateButtons() {
      // Update bet button states based on current bet states
      Object.keys(betCardStates).forEach(betId => {
          const betState = betCardStates[betId];
          const panelIndex = betId === 'bet1' ? 0 : 1;
          if (betState.state === 'bet') {
              updateBetButtonState(panelIndex, 'bet');
          } else {
              updateBetButtonState(panelIndex, betState.state);
          }
      });
  }

  // --- COUNTDOWN TIMER ---
  function startCountdown(ms) {
      stopCountdown();
      
      const endTime = Date.now() + ms;
      
      countdownInterval = setInterval(() => {
          const remaining = Math.max(0, endTime - Date.now());
          const seconds = Math.ceil(remaining / 1000);
          
          if (countdownDisplay) {
              countdownDisplay.textContent = `Next round in ${seconds}s`;
          }
          
          if (remaining <= 0) {
              stopCountdown();
          }
      }, 100);
  }

  function stopCountdown() {
      if (countdownInterval) {
          clearInterval(countdownInterval);
          countdownInterval = null;
      }
      if (countdownDisplay) {
          countdownDisplay.textContent = '';
      }
  }

  // --- BETTING FUNCTIONS ---
  function placeBet(amount, panelIndex) {
      if (currentGameState === GAME_STATES.PLAYING) {
          // If game is already flying, queue the bet for next round
          const betId = panelIndex === 0 ? 'bet1' : 'bet2';
          
          if (amount > userBalance) {
              showToast('Insufficient balance!');
              return false;
          }
          
          // Simulate WebSocket message
          const betMessage = { action: 'QUEUE_BET', amount: amount };
          console.log('Queueing bet for next round:', betMessage);
          
          // Update local state - deduct balance immediately (will sync with server when round starts)
          userBalance -= amount;
          betCardStates[betId] = {
              state: 'queued',
              betAmount: amount,
              cashoutValue: 0,
              cashedOut: false,
              isActive: false
          };
          
          updateBalanceDisplay();
          updateBetButtonState(panelIndex, 'queued');
          showToast(`Bet queued for next round: ${amount.toFixed(2)} R$`);
          
          return true;
      }
      
      if (currentGameState !== GAME_STATES.WAITING) {
          showToast('Betting is closed—please wait for the next round.');
          return false;
      }
      
      if (amount > userBalance) {
          showToast('Insufficient balance!');
          return false;
      }
      
      const betId = panelIndex === 0 ? 'bet1' : 'bet2';
      
      // Simulate WebSocket message
      const betMessage = { action: 'PLACE_BET', amount: amount };
      console.log('Sending bet:', betMessage);
      
      // Update local state (will sync with server when round starts)
      userBalance -= amount;
      betCardStates[betId] = {
          state: 'pending',
          betAmount: amount,
          cashoutValue: 0,
          cashedOut: false,
          isActive: false
      };
      
      updateBalanceDisplay();
      showToast(`Bet placed: ${amount.toFixed(2)} R$`);
      
      return true;
  }

  // Track active notifications for stacking
  let activeNotifications = [];
  
  function showCashoutNotification(multiplier, winAmount, currency = 'ZAR', notificationId = null) {
    const container = document.getElementById('notification-container');
    if (!container) return;
    
    // Create unique ID for this notification
    const id = notificationId || 'notification-' + Date.now() + '-' + Math.random().toString(36).substr(2, 9);
    
    // Create notification element
    const notification = document.createElement('div');
    notification.className = 'notification';
    notification.id = id;
    notification.style.display = 'none';
    
    notification.innerHTML = `
      <div class="notification__left">
        <div class="title">You have cashed out!</div>
        <div class="multiplier">${multiplier.toFixed(2)}x</div>
      </div>
      <div class="notification__pill">
        <div class="stars">★  ★</div>
        <div class="label">Win ${currency}</div>
        <div class="win">${winAmount.toFixed(2)}</div>
      </div>
      <div class="notification__close" onclick="hideNotification('${id}')">×</div>
    `;
    
    container.appendChild(notification);
    
    // Calculate position based on existing visible notifications
    const visibleNotifications = Array.from(container.querySelectorAll('.notification'))
      .filter(n => {
        const style = window.getComputedStyle(n);
        return style.display !== 'none' && (n.classList.contains('show') || style.top !== '-200px');
      });
    
    const index = visibleNotifications.length;
    const topOffset = 20 + (index * 90); // 20px from top, 90px spacing between notifications (70px height + 20px gap)
    
    notification.style.top = `${-200}px`; // Start above viewport
    notification.style.display = 'flex';
    
    // Update active notifications array
    activeNotifications.push(id);
    
    // Force reflow and animate in
    void notification.offsetWidth;
    
    // Use requestAnimationFrame for smoother animation
    requestAnimationFrame(() => {
      setTimeout(() => {
        notification.classList.add('show');
        notification.style.top = `${topOffset}px`;
      }, 10);
    });
    
    // Auto-hide after 3 seconds - store timeout ID
    const hideTimeoutId = setTimeout(() => {
      console.log('Auto-hiding notification:', id);
      hideNotification(id);
    }, 3000);
    
    // Store timeout ID on notification for cleanup
    notification.dataset.hideTimeoutId = hideTimeoutId.toString();
    
    // Debug: Log notification creation
    console.log('Notification created:', id, 'Will auto-hide in 3 seconds');
  }
  
  function hideNotification(notificationId) {
    if (!notificationId) {
      console.warn('hideNotification called without notificationId');
      return;
    }
    
    console.log('Hiding notification:', notificationId);
    
    const notification = document.getElementById(notificationId);
    if (!notification) {
      console.warn('Notification not found:', notificationId);
      return;
    }
    
    // Clear any pending hide timeout
    if (notification.dataset.hideTimeoutId) {
      clearTimeout(parseInt(notification.dataset.hideTimeoutId));
      delete notification.dataset.hideTimeoutId;
    }
    
    // Animate out - remove show class and move to top
    notification.classList.remove('show');
    
    // Force immediate style update
    requestAnimationFrame(() => {
      notification.style.top = `${-200}px`;
      notification.style.opacity = '0';
    });
    
    // Remove from DOM after animation completes (500ms for transition)
    const removeTimeout = setTimeout(() => {
      const stillExists = document.getElementById(notificationId);
      if (stillExists && stillExists.parentNode) {
        try {
          console.log('Removing notification from DOM:', notificationId);
          stillExists.parentNode.removeChild(stillExists);
        } catch (e) {
          console.log('Notification already removed:', e);
        }
      }
      
      // Remove from active notifications array
      activeNotifications = activeNotifications.filter(id => id !== notificationId);
      
      // Reposition remaining notifications
      repositionNotifications();
    }, 500);
    
    // Store remove timeout for potential cleanup
    notification.dataset.removeTimeoutId = removeTimeout.toString();
  }
  
  // Make hideNotification available globally for onclick handlers
  window.hideNotification = hideNotification;
  
  function repositionNotifications() {
    const container = document.getElementById('notification-container');
    if (!container) return;
    
    const visibleNotifications = Array.from(container.querySelectorAll('.notification'))
      .filter(n => {
        const style = window.getComputedStyle(n);
        return style.display !== 'none' && (n.classList.contains('show') || parseFloat(style.top) > -200);
      });
    
    visibleNotifications.forEach((notification, index) => {
      const topOffset = 20 + (index * 90); // 20px from top, 90px spacing between notifications
      notification.style.top = `${topOffset}px`;
    });
  }

  // In cashOut, after a successful cashout, show the notification
  function cashOut(panelIndex) {
      if (currentGameState !== GAME_STATES.PLAYING) {
          return false;
      }
      
      const betId = panelIndex === 0 ? 'bet1' : 'bet2';
      const betState = betCardStates[betId];
      
      if (betState.state !== 'cashout' || !betState.isActive) {
          return false;
      }
      
      // Simulate WebSocket message
      const cashoutMessage = { action: 'CASH_OUT' };
      console.log('Sending cashout:', cashoutMessage);
      
      // Calculate winnings using current multiplier
      const winnings = betState.betAmount * currentMultiplier;
      
      // Update round data for cashout (track total win amount)
      if (!currentRoundData.userWinAmount) {
          currentRoundData.userWinAmount = 0;
      }
      currentRoundData.userWinAmount += winnings;
      currentRoundData.userResult = 'win';
      
      // Update balance via API
      fetch('/api/balance/add', {
          method: 'POST',
          headers: {
              'Content-Type': 'application/json',
              'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || ''
          },
          body: JSON.stringify({
              amount: winnings,
              game_uuid: gameUuid
          })
      })
      .then(response => response.json())
      .then(data => {
          if (data.success) {
              userBalance = parseFloat(data.balance || userBalance + winnings);
              window.userBalance = userBalance;
              updateBalanceDisplay();
          }
      })
      .catch(error => {
          console.error('Failed to update balance:', error);
          userBalance += winnings;
          window.userBalance = userBalance;
          updateBalanceDisplay();
      });
      
      // Show the cashout notification
      showCashoutNotification(currentMultiplier, winnings, 'R$');
      
      // Update bet state - show cashed out state and amount
      betCardStates[betId] = {
          state: 'cashedout',
          betAmount: betState.betAmount,
          cashoutValue: winnings,
          cashedOut: true,
          isActive: false
      };
      
      updateBalanceDisplay();
      showToast(`Cashed out! Won: ${winnings.toFixed(2)} ZAR`);
      
      // Update button to show cashed out state and amount
      updateBetButtonState(panelIndex, 'cashedout');
      
      // Reset to default after 2 seconds
      setTimeout(() => {
          // Use the stored original amount
          const originalAmount = originalBetAmounts[betId];
          
          betCardStates[betId] = {
              state: 'bet',
              betAmount: originalAmount,
              cashoutValue: 0,
              cashedOut: false,
              isActive: false
          };
          updateBetButtonState(panelIndex, 'bet');
          updateBetControlsState(panelIndex, false);
          
          // If auto bet is enabled, place bet automatically for next round
          if (autoSettings[betId].autoBetEnabled && currentGameState === GAME_STATES.WAITING) {
              if (originalAmount <= userBalance) {
                  if (placeBet(originalAmount, panelIndex)) {
                      updateBetButtonState(panelIndex, 'pending');
                      updateBetControlsState(panelIndex, true);
                  }
              }
          }
      }, 2000);
      
      return true;
  }

  // --- UI FEEDBACK ---
  function showToast(message, duration = 3000) {
      // Create or find existing toast
      let toast = document.querySelector('.toast');
      if (!toast) {
          toast = document.createElement('div');
          toast.className = 'toast';
          toast.style.cssText = `
              position: fixed;
              top: 20px;
              right: 20px;
              background: #333;
              color: white;
              padding: 12px 20px;
              border-radius: 4px;
              z-index: 1000;
              font-family: Arial, sans-serif;
              font-size: 14px;
              opacity: 0;
              transition: opacity 0.3s;
          `;
          document.body.appendChild(toast);
      }
      
      toast.textContent = message;
      toast.style.opacity = '1';
      
      setTimeout(() => {
          toast.style.opacity = '0';
      }, duration);
  }

  function showGameResults(payload) {
      const multiplier = payload.finalMultiplier || 0;
      
      // Check if any bets were lost
      Object.keys(betCardStates).forEach(betId => {
          const betState = betCardStates[betId];
          if (betState.state === 'cashout' && !betState.cashedOut) {
              const message = `Plane flew away at ${multiplier.toFixed(2)}x - You lost ${betState.betAmount.toFixed(2)} ZAR`;
              showToast(message, 5000);
          }
      });
  }

  // --- AUTO BET ---
  function toggleAutoBet() {
      autoBetEnabled = !autoBetEnabled;
      if (autoBetToggle) {
          autoBetToggle.textContent = autoBetEnabled ? 'Auto Bet: ON' : 'Auto Bet: OFF';
          autoBetToggle.style.backgroundColor = autoBetEnabled ? '#10b981' : '#666';
      }
  }

  // --- ROUND STATE INTEGRATION ---
  // When round starts, move all pending bets to cashout state and deduct from balance if not already deducted
  function activatePendingBets() {
      Object.keys(betCardStates).forEach(betId => {
          const state = betCardStates[betId];
          if (state.state === 'pending') {
              // Only deduct if not already deducted (i.e., if bet was queued during PLAYING)
              if (currentGameState === 'WAITING') {
                  if (userBalance < state.betAmount) {
                      showToast('Insufficient balance for queued bet!');
                      state.state = 'bet';
                      updateBetCardButton(betId);
                      return;
                  }
                  userBalance -= state.betAmount;
                  updateBalanceDisplay();
              }
              state.state = 'cashout';
              state.cashoutValue = state.betAmount;
              updateBetCardButton(betId);
          }
      });
  }

  // --- EVENT HANDLERS ---
  if (betButton) {
      betButton.addEventListener('click', () => {
          if (currentGameState === GAME_STATES.WAITING) {
              const amount = getCurrentBetAmount();
              placeBet(amount, 0);
          } else if (currentGameState === GAME_STATES.PLAYING && userHasActiveBet) {
              cashOut(0);
          } else {
              showToast('Action not available in current state.');
          }
      });
  }

  if (autoBetToggle) {
      autoBetToggle.addEventListener('click', toggleAutoBet);
  }

  // --- SIMULATED WEBSOCKET EVENTS ---
  function simulateWebSocketEvents() {
      // Simulate game start after 3 seconds
      setTimeout(() => {
          onGameStateChange(GAME_STATES.PLAYING, { 
              roundId: 'round_' + Date.now(),
              startTimestamp: Date.now()
          });
      }, 3000);
  }

  // --- INTEGRATION WITH EXISTING GAME CYCLE ---
  function startPlaneAnimation() {
      // Your existing plane animation logic here
      // This should trigger when entering PLAYING state
      console.log('Starting plane animation for round:', currentRoundId);
      
      // Simulate game end after random time (5-15 seconds)
      const gameDuration = Math.random() * 10000 + 5000;
      setTimeout(() => {
          const finalMultiplier = (Math.random() * 8 + 1.2).toFixed(2);
          onGameStateChange(GAME_STATES.ENDED, { finalMultiplier: parseFloat(finalMultiplier) });
      }, gameDuration);
  }

  // --- INITIALIZATION ---
  function initializeGameState() {
      // Force initial value for both .bet-amount and bet button
      document.querySelectorAll('.bet-amount').forEach((amtEl, index) => {
          amtEl.textContent = '2.00';
          const betBtnAmount = document.querySelectorAll('.bet-btn-amount')[index];
          if (betBtnAmount) {
              betBtnAmount.textContent = '2.00 ZAR';
          }
      });
      // Initialize UI
      updateButtons();
      updateBalanceDisplay();
      
      // Initialize dynamic betting stats
      updateBettingStatsDisplay();
      
      // Initialize players indicator
      updatePlayersIndicator();
  }

  // Sidebar Tab Switching
  function initializeSidebarTabs() {
    const tabs = document.querySelectorAll('.sidebar .tabs .tab');
    const tabContents = document.querySelectorAll('.sidebar .tab-content');
    
    tabs.forEach(tab => {
      tab.addEventListener('click', function() {
        const targetTab = this.getAttribute('data-tab');
        
        // Remove active class from all tabs and contents
        tabs.forEach(t => t.classList.remove('active'));
        tabContents.forEach(content => content.classList.remove('active'));
        
        // Add active class to clicked tab and corresponding content
        this.classList.add('active');
        const targetContent = document.getElementById(targetTab + '-content');
        if (targetContent) {
          targetContent.classList.add('active');
          
          // Load data for the active tab
          if (targetTab === 'all-bets') {
            initializeAllBetsTab();
          } else if (targetTab === 'previous') {
            loadPreviousRounds();
          } else if (targetTab === 'top') {
            loadTopWinners();
          }
        }
      });
    });
    
    // Initialize All Bets tab on load
    initializeAllBetsTab();
  }
  
  function initializeAllBetsTab() {
    const betsList = document.getElementById('all-bets-list');
    if (!betsList) return;
    
    // Don't clear if already populated to avoid flickering
    if (betsList.children.length === 0) {
      // Add some initial bet items only if list is empty
      const initialBets = [
        { player: '3***5', bet: '45.20', multiplier: '2.45x', win: '110.74', won: true },
        { player: '7***2', bet: '12.50', multiplier: '1.89x', win: '23.63', won: true },
        { player: '1***8', bet: '88.00', multiplier: '3.21x', win: '282.48', won: true },
        { player: '9***4', bet: '25.00', multiplier: '1.12x', win: '', won: false },
        { player: '5***1', bet: '150.00', multiplier: '5.67x', win: '850.50', won: true },
        { player: '2***7', bet: '30.00', multiplier: '1.05x', win: '', won: false },
        { player: '4***3', bet: '67.50', multiplier: '2.88x', win: '194.40', won: true },
      ];
      
      initialBets.forEach((bet, index) => {
        const betItem = document.createElement('div');
        betItem.className = `bet-item ${bet.won ? 'won' : ''}`;
        betItem.innerHTML = `
          <div class="bet-player">
            <div class="avatar" style="background-image: url('https://i.pravatar.cc/24?u=${bet.player}')"></div>
            <span>${bet.player}</span>
          </div>
          <div class="bet-amount">${bet.bet}</div>
          <div class="bet-multiplier ${bet.won ? 'green' : 'blue'}">${bet.multiplier}</div>
          <div class="bet-win">${bet.win || '—'}</div>
        `;
        betItem.style.opacity = '0';
        betItem.style.transform = 'translateY(-10px)';
        betsList.appendChild(betItem);
        
        setTimeout(() => {
          betItem.style.transition = 'opacity 0.3s, transform 0.3s';
          betItem.style.opacity = '1';
          betItem.style.transform = 'translateY(0)';
        }, index * 50);
      });
    }
  }
  
  async function loadPreviousRounds() {
    const previousList = document.getElementById('previous-rounds-list');
    const roundResultMultiplier = document.getElementById('round-result-multiplier');
    if (!previousList) return;
    
    try {
      const response = await fetch('/api/flew-away-multipliers?limit=1', {
        headers: {
          'X-Requested-With': 'XMLHttpRequest',
          'Accept': 'application/json',
        },
        credentials: 'same-origin',
      });
      
      let roundMultiplier = '2.24x';
      if (response.ok) {
        const data = await response.json();
        if (data.success && data.multipliers && data.multipliers.length > 0) {
          roundMultiplier = parseFloat(data.multipliers[0].multiplier).toFixed(2) + 'x';
        }
      }
      
      // Update round result header
      if (roundResultMultiplier) {
        roundResultMultiplier.textContent = roundMultiplier;
      }
      
      // Generate sample bet data for the previous round
      previousList.innerHTML = '';
      const sampleBets = [
        { player: '2***8', bet: '1,450.00', multiplier: '', win: '0.00', won: false },
        { player: 'n***j', bet: '550.00', multiplier: '', win: '0.00', won: false },
        { player: '5***5', bet: '550.00', multiplier: '', win: '0.00', won: false },
        { player: 'g***u', bet: '500.00', multiplier: '', win: '0.00', won: false },
        { player: '5***2', bet: '500.00', multiplier: '2.11x', win: '1,055.00', won: true },
        { player: '5***1', bet: '500.00', multiplier: '2.04x', win: '1,020.00', won: true },
        { player: '2***8', bet: '500.00', multiplier: '', win: '0.00', won: false },
        { player: '5***2', bet: '500.00', multiplier: '', win: '0.00', won: false },
        { player: 'm***e', bet: '423.66', multiplier: '2.04x', win: '864.27', won: true },
        { player: 'm***e', bet: '423.66', multiplier: '', win: '0.00', won: false },
        { player: '0***8', bet: '400.00', multiplier: '2.14x', win: '856.00', won: true },
        { player: '6***8', bet: '320.00', multiplier: '1.63x', win: '521.60', won: true },
        { player: '5***8', bet: '300.00', multiplier: '', win: '0.00', won: false },
        { player: 'w***U', bet: '300.00', multiplier: '2.00x', win: '600.00', won: true },
        { player: 'g***u', bet: '300.00', multiplier: '2.04x', win: '612.00', won: true },
        { player: '5***7', bet: '280.51', multiplier: '', win: '0.00', won: false },
        { player: '6***8', bet: '280.00', multiplier: '1.24x', win: '347.20', won: true },
        { player: '2***1', bet: '250.00', multiplier: '', win: '0.00', won: false },
        { player: '5***1', bet: '200.00', multiplier: '', win: '0.00', won: false },
        { player: '5***5', bet: '200.00', multiplier: '1.71x', win: '342.00', won: true },
        { player: '2***6', bet: '200.00', multiplier: '2.14x', win: '428.00', won: true },
        { player: 'n***7', bet: '200.00', multiplier: '', win: '0.00', won: false },
        { player: 'n***y', bet: '200.00', multiplier: '', win: '0.00', won: false },
        { player: 'n***y', bet: '200.00', multiplier: '', win: '0.00', won: false },
        { player: '2***1', bet: '200.00', multiplier: '', win: '0.00', won: false },
        { player: 'k***2', bet: '200.00', multiplier: '', win: '0.00', won: false },
        { player: '0***6', bet: '190.64', multiplier: '', win: '0.00', won: false },
      ];
      
      sampleBets.forEach((bet, index) => {
        const betItem = document.createElement('div');
        betItem.className = `previous-bet-item ${bet.won ? 'won' : ''}`;
        const randomUser = bet.player.replace(/\*/g, '');
        betItem.innerHTML = `
          <div class="previous-bet-player">
            <div class="avatar" style="background-image: url('https://i.pravatar.cc/24?u=${randomUser}')"></div>
            <span>${bet.player}</span>
          </div>
          <div class="previous-bet-amount">${bet.bet}</div>
          <div class="previous-bet-multiplier">${bet.multiplier || '—'}</div>
          <div class="previous-bet-win">${bet.win}</div>
        `;
        
        betItem.style.opacity = '0';
        betItem.style.transform = 'translateY(-5px)';
        previousList.appendChild(betItem);
        
        setTimeout(() => {
          betItem.style.transition = 'opacity 0.3s, transform 0.3s';
          betItem.style.opacity = '1';
          betItem.style.transform = 'translateY(0)';
        }, index * 20);
      });
    } catch (error) {
      console.error('Error loading previous rounds:', error);
      previousList.innerHTML = '<div style="text-align: center; padding: 2rem; color: #888;">Error loading data</div>';
    }
  }
  
  async function loadTopWinners() {
    const topList = document.getElementById('top-winners-list');
    if (!topList) return;
    
    try {
      const response = await fetch('/api/history?filter=win', {
        headers: {
          'X-Requested-With': 'XMLHttpRequest',
          'Accept': 'application/json',
        },
        credentials: 'same-origin',
      });
      
      topList.innerHTML = '';
      
      // Generate sample top winners matching the image format
      const sampleWinners = [
        { username: 'a***e', bet: '1.00', win: '1,511.11', result: '1,511.11x', roundMax: '1,691.62x', avatar: 'a' },
        { username: 'n***1', bet: '19.50', win: '24,548.55', result: '1,258.90x', roundMax: '1,691.62x', avatar: 'n' },
        { username: '5***4', bet: '0.10', win: '124.85', result: '1,248.50x', roundMax: '1,691.62x', avatar: '5' },
        { username: '2***2', bet: '1.00', win: '1,227.94', result: '1,227.94x', roundMax: '1,691.62x', avatar: '2' },
        { username: '0***a', bet: '0.42', win: '490.86', result: '1,158.63x', roundMax: '1,691.62x', avatar: '0' },
        { username: 'e***e', bet: '0.50', win: '555.76', result: '1,111.53x', roundMax: '1,691.62x', avatar: 'e' },
        { username: '5***4', bet: '3.00', win: '3,279.69', result: '1,093.23x', roundMax: '1,691.62x', avatar: '54' },
        { username: '5***9', bet: '0.10', win: '107.52', result: '1,075.24x', roundMax: '1,691.62x', avatar: '59' },
        { username: '0***4', bet: '2.00', win: '2,132.70', result: '1,066.35x', roundMax: '1,691.62x', avatar: '04' },
        { username: '5***4', bet: '1.00', win: '1,066.35', result: '1,066.35x', roundMax: '1,691.62x', avatar: '54b' },
      ];
      
      const today = new Date();
      const dateStr = `${String(today.getDate()).padStart(2, '0')}.${String(today.getMonth() + 1).padStart(2, '0')}.${String(today.getFullYear()).slice(-2)}`;
      
      sampleWinners.forEach((winner, index) => {
        const winnerItem = document.createElement('div');
        winnerItem.className = 'top-winner-item';
        winnerItem.innerHTML = `
          <div class="top-winner-avatar" style="background-image: url('https://i.pravatar.cc/40?u=${winner.avatar}')"></div>
          <div class="top-winner-info">
            <div class="top-winner-username">${winner.username}</div>
            <div class="top-winner-date">${dateStr}</div>
            <div class="top-winner-amounts">
              <div class="top-winner-bet-row">
                <span class="top-winner-bet-label">Bet R$</span>
                <span class="top-winner-bet-value">${winner.bet}</span>
              </div>
              <div class="top-winner-win-row">
                <span class="top-winner-win-label">Win R$</span>
                <span class="top-winner-win-value">${winner.win}</span>
              </div>
            </div>
          </div>
          <div class="top-winner-multipliers">
            <div class="top-winner-result-multiplier">${winner.result}</div>
            <div class="top-winner-round-max">Round max. ${winner.roundMax}</div>
          </div>
          <div class="top-winner-actions">
            <button class="top-winner-action-btn" title="Chat">
              <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"/>
              </svg>
            </button>
            <button class="top-winner-action-btn" title="Verify">
              <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M12 22s8-4 8-10V5l-8-3-8 3v7c0 6 8 10 8 10z"/>
                <polyline points="9 12 11 14 15 10"/>
              </svg>
            </button>
          </div>
        `;
        
        winnerItem.style.opacity = '0';
        winnerItem.style.transform = 'translateY(-10px)';
        topList.appendChild(winnerItem);
        
        setTimeout(() => {
          winnerItem.style.transition = 'opacity 0.3s, transform 0.3s';
          winnerItem.style.opacity = '1';
          winnerItem.style.transform = 'translateY(0)';
        }, index * 50);
      });
      
      // Add filter button handlers
      const filterBtns = document.querySelectorAll('.filter-btn');
      filterBtns.forEach(btn => {
        btn.addEventListener('click', function() {
          const filterRow = this.closest('.filter-row');
          filterRow.querySelectorAll('.filter-btn').forEach(b => b.classList.remove('active'));
          this.classList.add('active');
          // Here you could reload data based on filter
        });
      });
    } catch (error) {
      console.error('Error loading top winners:', error);
      topList.innerHTML = '<div style="text-align: center; padding: 2rem; color: #888;">Error loading data</div>';
    }
  }

  // Initialize when DOM is ready
  document.addEventListener('DOMContentLoaded', function() {
      // Load flew away multipliers from database first
      loadFlewAwayMultipliers().then(() => {
          initializeGameState();
          
          // Initialize the new betting interface
          initializeBettingInterface();
          
          // Initialize sidebar tabs
          initializeSidebarTabs();
      });
  });

  // Initialize the new betting interface
  function initializeBettingInterface() {
    // Initialize Bet/Auto tab switching and button handlers for each panel
    document.querySelectorAll('.betting-panel').forEach((panel, panelIndex) => {
      // Ensure panelIndex is correctly identified from data-panel-index attribute or position
      const panelDataIndex = panel.getAttribute('data-panel-index');
      const actualPanelIndex = panelDataIndex !== null ? parseInt(panelDataIndex) : panelIndex;
      const tabs = panel.querySelectorAll('.tab-container .tab');
      const betContent = panel.querySelector('.bet-content');
      const autoContent = panel.querySelector('.auto-content');
      
      tabs.forEach(tab => {
        tab.addEventListener('click', function() {
          // Remove active from all tabs in this panel
          tabs.forEach(t => {
            t.classList.remove('active');
            t.classList.add('inactive');
          });
          
          // Add active to clicked tab
          this.classList.remove('inactive');
          this.classList.add('active');
          
          // Show/hide content
          const tabType = this.getAttribute('data-tab');
          if (tabType === 'bet') {
            // Remove active from auto, add to bet
            autoContent.classList.remove('active');
            betContent.classList.add('active');
            // Sync bet amounts when switching to bet tab
            syncBetAmountDisplay(actualPanelIndex);
          } else if (tabType === 'auto') {
            // Remove active from bet, add to auto
            betContent.classList.remove('active');
            autoContent.classList.add('active');
            // Sync bet amounts when switching to auto tab
            syncBetAmountDisplay(actualPanelIndex);
          }
          
          // Force reflow to ensure CSS changes take effect
          void betContent.offsetHeight;
          void autoContent.offsetHeight;
        });
      });
      
      // Initialize auto bet toggle - use data-panel attribute to ensure correct scoping
      const autoBetToggle = panel.querySelector('.auto-bet-toggle[data-panel="' + actualPanelIndex + '"]');
      if (autoBetToggle) {
        autoBetToggle.addEventListener('change', function() {
          const betId = actualPanelIndex === 0 ? 'bet1' : 'bet2';
          autoSettings[betId].autoBetEnabled = this.checked;
          
          // If auto bet is enabled and we're in WAITING state, place bet automatically
          if (this.checked && currentGameState === GAME_STATES.WAITING) {
            const betAmount = betCardStates[betId].betAmount;
            if (betAmount <= userBalance) {
              if (placeBet(betAmount, actualPanelIndex)) {
                updateBetButtonState(actualPanelIndex, 'pending');
                updateBetControlsState(actualPanelIndex, true);
              }
            }
          }
        });
      }
      
      // Initialize auto cashout toggle - use data-panel attribute to ensure correct scoping
      const autoCashoutToggle = panel.querySelector('.auto-cashout-toggle[data-panel="' + actualPanelIndex + '"]');
      const autoCashoutInput = panel.querySelector('.auto-cashout-input[data-panel="' + actualPanelIndex + '"]');
      if (autoCashoutToggle && autoCashoutInput) {
        // Initialize multiplier value from settings
        const betId = actualPanelIndex === 0 ? 'bet1' : 'bet2';
        autoCashoutInput.value = autoSettings[betId].autoCashoutMultiplier.toFixed(2);
        
        // Ensure input wrapper is always visible
        const inputWrapper = autoCashoutInput.closest('.auto-cashout-input-wrapper');
        if (inputWrapper) {
          inputWrapper.style.display = 'flex';
          if (!autoCashoutToggle.checked) {
            inputWrapper.style.opacity = '0.6';
          }
        }
        
        autoCashoutToggle.addEventListener('change', function() {
          const betId = actualPanelIndex === 0 ? 'bet1' : 'bet2';
          autoSettings[betId].autoCashoutEnabled = this.checked;
          
          // Input field is always visible, adjust opacity based on toggle state
          const inputWrapper = autoCashoutInput.closest('.auto-cashout-input-wrapper');
          if (this.checked && inputWrapper) {
            inputWrapper.style.opacity = '1';
            setTimeout(() => autoCashoutInput.focus(), 100);
          } else if (inputWrapper) {
            inputWrapper.style.opacity = '0.6';
          }
        });
        
        // Update multiplier when input changes
        autoCashoutInput.addEventListener('input', function() {
          const betId = actualPanelIndex === 0 ? 'bet1' : 'bet2';
          const multiplier = parseFloat(this.value) || 1.10;
          autoSettings[betId].autoCashoutMultiplier = Math.max(1.01, multiplier);
        });
        
        // Clear button functionality
        const clearBtn = panel.querySelector('.auto-cashout-clear[data-panel="' + actualPanelIndex + '"]');
        if (clearBtn) {
          clearBtn.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            const betId = actualPanelIndex === 0 ? 'bet1' : 'bet2';
            autoCashoutInput.value = '1.10';
            autoSettings[betId].autoCashoutMultiplier = 1.10;
            autoCashoutInput.focus();
          });
        }
      }
    });
    
    // Add click handlers for main bet buttons - scoped to each panel
    // Note: onclick handlers in HTML already handle clicks correctly with panelIndex
    // We don't need additional listeners since onclick is working, but we ensure they're isolated
    document.querySelectorAll('.betting-panel').forEach((panel, panelIndex) => {
      // Ensure panelIndex is correctly identified
      const panelDataIndex = panel.getAttribute('data-panel-index');
      const actualPanelIndex = panelDataIndex !== null ? parseInt(panelDataIndex) : panelIndex;
      
      // Verify onclick handlers are using correct index
      const betBtns = panel.querySelectorAll('.main-bet-btn');
      betBtns.forEach((btn) => {
        // Ensure onclick attribute uses correct panelIndex
        const onclickAttr = btn.getAttribute('onclick');
        if (onclickAttr && onclickAttr.includes('handleBetButtonClick')) {
          // Verify it's using the correct index
          const match = onclickAttr.match(/handleBetButtonClick\((\d+)\)/);
          if (match && parseInt(match[1]) !== actualPanelIndex) {
            // Fix incorrect index
            btn.setAttribute('onclick', `handleBetButtonClick(${actualPanelIndex})`);
          }
        }
      });
    });
  }

  // Handle main bet button clicks
  function handleBetButtonClick(panelIndex) {
    const betId = panelIndex === 0 ? 'bet1' : 'bet2';
    const betState = betCardStates[betId];
    
    if (!betState) {
      console.error('Bet state not found for panel:', panelIndex);
      return;
    }
    
    if (currentGameState === GAME_STATES.WAITING) {
      // If pending, clicking again cancels the bet
      if (betState.state === 'pending') {
        // Cancel pending bet - refund balance
        userBalance += betState.betAmount;
        updateBalanceDisplay();
        
        // Reset to original bet amount
        const originalAmount = originalBetAmounts[betId];
        betCardStates[betId] = {
          state: 'bet',
          betAmount: originalAmount,
          cashoutValue: 0,
          cashedOut: false,
          isActive: false
        };
        updateBetButtonState(panelIndex, 'bet');
        updateBetControlsState(panelIndex, false);
        showToast('Bet cancelled and refunded.');
        return;
      }
      
      // If queued, pressing cancels the queued bet
      if (betState.state === 'queued') {
        // Refund the queued amount
        userBalance += betState.betAmount;
        updateBalanceDisplay();
        
        // Reset to original bet amount
        const originalAmount = originalBetAmounts[betId];
        betCardStates[betId] = {
          state: 'bet',
          betAmount: originalAmount,
          cashoutValue: 0,
          cashedOut: false,
          isActive: false
        };
        updateBetButtonState(panelIndex, 'bet');
        updateBetControlsState(panelIndex, false);
        showToast('Queued bet cancelled and refunded.');
        return;
      }
      
      // Otherwise, place bet using the amount from betCardStates
      const actualBetAmount = betState.betAmount;
      if (placeBet(actualBetAmount, panelIndex)) {
        updateBetButtonState(panelIndex, 'pending');
        updateBetControlsState(panelIndex, true);
      }
    } else if (currentGameState === GAME_STATES.PLAYING) {
      // If bet is active and can cash out, cash out
      if (betState.state === 'cashout' && betState.isActive) {
        cashOut(panelIndex);
        return;
      }
      
      // If bet is queued (waiting for next round), cancel it
      if (betState.state === 'queued') {
        // Refund the queued amount
        userBalance += betState.betAmount;
        updateBalanceDisplay();
        
        // Reset to original bet amount
        const originalAmount = originalBetAmounts[betId];
        betCardStates[betId] = {
          state: 'bet',
          betAmount: originalAmount,
          cashoutValue: 0,
          cashedOut: false,
          isActive: false
        };
        updateBetButtonState(panelIndex, 'bet');
        updateBetControlsState(panelIndex, false);
        showToast('Queued bet cancelled and refunded.');
        return;
      }
      
      // If no active bet, queue for next round
      if (betState.state === 'bet' || betState.state === 'cashedout' || betState.state === 'lost') {
        const actualBetAmount = betState.betAmount;
        if (placeBet(actualBetAmount, panelIndex)) {
          // Button state is updated in placeBet (sets to 'queued')
          updateBetControlsState(panelIndex, true);
        }
      }
    }
  }
  
  // Make handleBetButtonClick global
  window.handleBetButtonClick = handleBetButtonClick;
  
  // Clear auto cashout input function
  function clearAutoCashoutInput(panelIndex) {
    const panel = document.querySelectorAll('.betting-panel')[panelIndex];
    if (!panel) return;
    
    const input = panel.querySelector('.auto-cashout-input[data-panel="' + panelIndex + '"]');
    const betId = panelIndex === 0 ? 'bet1' : 'bet2';
    
    if (input) {
      input.value = '1.10';
      autoSettings[betId].autoCashoutMultiplier = 1.10;
      input.focus();
    }
  }
  
  window.clearAutoCashoutInput = clearAutoCashoutInput;

  // Update bet controls state (enable/disable)
  function updateBetControlsState(panelIndex, isBetActive) {
    const panel = document.querySelectorAll('.betting-panel')[panelIndex];
    if (!panel) return;
    
    // Update controls in both bet-content and auto-content
    if (isBetActive) {
      panel.classList.add('bet-active');
      // Disable all controls in both sections
      panel.querySelectorAll('.control-btn, .quick-bet').forEach(btn => {
        btn.style.pointerEvents = 'none';
        btn.style.opacity = '0.5';
      });
    } else {
      panel.classList.remove('bet-active');
      // Enable all controls in both sections
      panel.querySelectorAll('.control-btn, .quick-bet').forEach(btn => {
        btn.style.pointerEvents = '';
        btn.style.opacity = '';
      });
    }
  }
  

  function updateMultiplier(multiplier) {
      if (multiplierDisplay) {
          multiplierDisplay.textContent = multiplier.toFixed(2) + 'x';
      }
      
      // Update cashout values for active bets
      updateBetCardCashoutValues(multiplier);
  }

  // Add this function near the top or with other UI helpers
  function addFlewAwayMultiplier(multiplier) {
    const multipliersDiv = document.querySelector('.multipliers');
    if (!multipliersDiv) return;
    const span = document.createElement('span');
    
    // Use random colors like the bet items
    const colors = ['blue','purple', 'red'];
    const randomColor = colors[Math.floor(Math.random() * colors.length)];
    
    span.className = `multiplier ${randomColor}`;
    span.textContent = multiplier.toFixed(2) + 'x';
    multipliersDiv.insertBefore(span, multipliersDiv.firstChild);
    // Optionally: limit to last 40 multipliers
    while (multipliersDiv.children.length > 40) {
      multipliersDiv.removeChild(multipliersDiv.lastChild);
    }
  }

  // Old showNotification function removed - using new notification system

  // Add this helper at the top-level (after betCardStates definition)
  function syncBetAmountDisplay(panelIndex) {
    const idx = panelIndex;
    if (amtEl[idx] && btnAmtEl[idx]) {
      const val = parseFloat(amtEl[idx].textContent) || 2.00;
      updateDisplay(idx, val);
    }
  }

  // Add this to script.js for compatibility with inline onclick="setBet(x)"
  function setBet(x) {
    // same logic to detect panel index
    const quickBets = document.querySelectorAll('.quick-bet');
    const idx = Math.floor(Array.from(quickBets).indexOf(event.currentTarget) / 4);
    if (idx >= 0) {
      updateDisplay(idx, x);
    }
  }

  // Dynamic betting stats update system
  let bettingStatsInterval;
  let bettingStatsAnimationId;
  let currentBettingStats = {
    currentBets: 536,
    totalBets: 669,
    totalWin: 1286.51
  };

  let targetBettingStats = {
    currentBets: 536,
    totalBets: 669,
    totalWin: 1286.51
  };

  // Flag to track if stats have been initialized
  let statsInitialized = false;

  // Dynamic players indicator system
  let currentPlayersCount = 235;
  let targetPlayersCount = 235;

  function startDynamicBettingStats() {
    // Clear any existing interval and animation
    if (bettingStatsInterval) {
      clearInterval(bettingStatsInterval);
    }
    if (bettingStatsAnimationId) {
      cancelAnimationFrame(bettingStatsAnimationId);
    }
    
    // Only initialize if this is the first time
    if (!statsInitialized) {
      // This is the first time, set initial values
      currentBettingStats = {
        currentBets: 536,
        totalBets: 669,
        totalWin: 1286.51
      };
      
      targetBettingStats = {
        currentBets: 536,
        totalBets: 669,
        totalWin: 1286.51
      };
      
      // Initialize players count
      currentPlayersCount = 235;
      targetPlayersCount = 235;
      
      statsInitialized = true;
    } else {
      // Continue from current values, just set targets to current
      targetBettingStats = { ...currentBettingStats };
      targetPlayersCount = currentPlayersCount;
    }
    
    updateBettingStatsDisplay();
    updatePlayersIndicator();
    
    // Start the dynamic updates with more frequent intervals
    bettingStatsInterval = setInterval(() => {
      // Randomly add 1-5 new bets for more dynamic feel
      const newBets = Math.floor(Math.random() * 5) + 1;
      targetBettingStats.currentBets += newBets;
      
      // Occasionally increase total bets (simulating new players joining)
      if (Math.random() < 0.4) {
        targetBettingStats.totalBets += Math.floor(Math.random() * 3) + 1;
      }
      
      // Increase total win amount more dynamically
      targetBettingStats.totalWin += (Math.random() * 100 + 20);
      
      // Update players count randomly (less frequently)
      if (Math.random() < 0.6) { // 60% chance to update player count
        const playerChange = Math.floor(Math.random() * 15) - 5; // -5 to +10
        targetPlayersCount = Math.max(200, Math.min(500, targetPlayersCount + playerChange));
        animatePlayersCount();
      }
      
      // Start smooth animation to target values
      animateBettingStats();
      
      // Add new bet items to the list
      addNewBetItems(newBets);
      
    }, Math.random() * 2000 + 1000); // Slower updates: 1-3 seconds
  }

  function stopDynamicBettingStats() {
    if (bettingStatsInterval) {
      clearInterval(bettingStatsInterval);
      bettingStatsInterval = null;
    }
    if (bettingStatsAnimationId) {
      cancelAnimationFrame(bettingStatsAnimationId);
      bettingStatsAnimationId = null;
    }
  }

  function animateBettingStats() {
    const duration = 1000; // 1 second animation
    const startTime = performance.now();
    
    function animate(currentTime) {
      const elapsed = currentTime - startTime;
      const progress = Math.min(elapsed / duration, 1);
      
      // Use ease-out function for smooth animation
      const easeOut = 1 - Math.pow(1 - progress, 3);
      
      // Interpolate between current and target values
      currentBettingStats.currentBets = Math.round(
        currentBettingStats.currentBets + (targetBettingStats.currentBets - currentBettingStats.currentBets) * easeOut
      );
      
      currentBettingStats.totalBets = Math.round(
        currentBettingStats.totalBets + (targetBettingStats.totalBets - currentBettingStats.totalBets) * easeOut
      );
      
      currentBettingStats.totalWin = 
        currentBettingStats.totalWin + (targetBettingStats.totalWin - currentBettingStats.totalWin) * easeOut;
      
      updateBettingStatsDisplay();
      
      if (progress < 1) {
        bettingStatsAnimationId = requestAnimationFrame(animate);
      }
    }
    
    bettingStatsAnimationId = requestAnimationFrame(animate);
  }

  function updateBettingStatsDisplay() {
    // Find stats within the active "all-bets" tab content
    const allBetsContent = document.getElementById('all-bets-content');
    const statsText = allBetsContent ? allBetsContent.querySelector('.stats-text') : document.querySelector('.stats-text');
    const statsProgress = allBetsContent ? allBetsContent.querySelector('.stats-progress-inner') : document.querySelector('.stats-progress-inner');
    const statsNumber = allBetsContent ? allBetsContent.querySelector('.stats-number') : document.querySelector('.stats-number');
    
    if (statsText) {
      statsText.textContent = `${currentBettingStats.currentBets}/${currentBettingStats.totalBets} Bets`;
    }
    
    if (statsProgress) {
      const progressPercentage = (currentBettingStats.currentBets / currentBettingStats.totalBets) * 100;
      // Add smooth transition to progress bar
      statsProgress.style.transition = 'width 0.3s ease-out';
      statsProgress.style.width = `${Math.min(progressPercentage, 100)}%`;
    }
    
    if (statsNumber) {
      // Add a subtle glow effect when the number changes
      statsNumber.style.textShadow = '0 0 8px rgba(18, 223, 56, 0.6)';
      statsNumber.textContent = currentBettingStats.totalWin.toFixed(2);
      
      // Remove the glow effect after a short delay
      setTimeout(() => {
        statsNumber.style.textShadow = 'none';
      }, 300);
    }
  }

  function addNewBetItems(count) {
    const betsList = document.getElementById('all-bets-list') || document.querySelector('.bets-list');
    if (!betsList) return;
    
    for (let i = 0; i < count; i++) {
      const randomUserId = `user${Math.random()}`;
      const playerName = `${Math.floor(Math.random() * 9) + 1}***${Math.floor(Math.random() * 9)}`;
      const betAmount = (Math.random() * 500 + 20).toFixed(2);
      
      const betItem = document.createElement('div');
      betItem.classList.add('bet-item');
      betItem.setAttribute('data-bet-amount', betAmount);
      
      betItem.innerHTML = `
        <div class="bet-player">
          <div class="avatar" style="background-image: url('https://i.pravatar.cc/24?u=${randomUserId}')"></div>
          <span>${playerName}</span>
        </div>
        <div class="bet-amount">${betAmount}</div>
        <div></div>
        <div></div>
      `;
      
      // Add with a more dramatic fade-in effect
      betItem.style.opacity = '0';
      betItem.style.transform = 'translateY(-20px) scale(0.9)';
      betItem.style.transition = 'opacity 0.6s ease-out, transform 0.6s ease-out';
      
      betsList.appendChild(betItem);
      
      // Trigger the animation with a slight delay for staggered effect
      setTimeout(() => {
        betItem.style.opacity = '1';
        betItem.style.transform = 'translateY(0) scale(1)';
      }, i * 100); // Stagger each item by 100ms
      
      // Remove old bet items if there are too many (keep max 25)
      const allBetItems = betsList.querySelectorAll('.bet-item');
      if (allBetItems.length > 25) {
        const oldestItem = allBetItems[0];
        oldestItem.style.opacity = '0';
        oldestItem.style.transform = 'translateY(10px) scale(0.9)';
        setTimeout(() => {
          if (oldestItem.parentNode) {
            oldestItem.parentNode.removeChild(oldestItem);
          }
        }, 600);
      }
    }
  }

  function updatePlayersIndicator() {
    const playersIndicator = document.querySelector('.players-indicator span');
    const playersIcon = document.querySelector('.players-indicator .players-icon');
    
    if (playersIndicator) {
      playersIndicator.textContent = currentPlayersCount;
    }
    
    // Only change avatars occasionally (30% chance) to slow down the changes
    if (playersIcon && Math.random() < 0.3) {
      // Clear existing avatars
      playersIcon.innerHTML = '';
      
      // Add 3 random avatars
      for (let i = 0; i < 3; i++) {
        const avatar = document.createElement('div');
        avatar.className = 'avatar';
        const randomUser = String.fromCharCode(97 + Math.floor(Math.random() * 26)); // a-z
        avatar.style.backgroundImage = `url('https://i.pravatar.cc/24?u=${randomUser}')`;
        playersIcon.appendChild(avatar);
      }
    }
  }

  function animatePlayersCount() {
    const duration = 800; // 0.8 second animation
    const startTime = performance.now();
    
    function animate(currentTime) {
      const elapsed = currentTime - startTime;
      const progress = Math.min(elapsed / duration, 1);
      
      // Use ease-out function for smooth animation
      const easeOut = 1 - Math.pow(1 - progress, 3);
      
      // Interpolate between current and target values
      currentPlayersCount = Math.round(
        currentPlayersCount + (targetPlayersCount - currentPlayersCount) * easeOut
      );
      
      updatePlayersIndicator();
      
      if (progress < 1) {
        requestAnimationFrame(animate);
      }
    }
    
    requestAnimationFrame(animate);
  }
