<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'last_name',
        'cpf',
        'phone',
        'avatar',
        'oauth_id',
        'oauth_type',
        'inviter',
        'inviter_code',
        'affiliate_revenue_share',
        'affiliate_cpa',
        'status',
        'language',
        'role_id',
        'banned',
        'last_activity',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'last_activity' => 'datetime',
        ];
    }

    /**
     * Get the user's orders.
     */
    public function orders()
    {
        return $this->hasMany(Order::class);
    }

    /**
     * Get the user's deposits.
     */
    public function deposits()
    {
        return $this->hasMany(Deposit::class);
    }

    /**
     * Calculate user balance from deposits and orders.
     * Balance = Total Deposits (approved) + Total Wins - Total Losses
     */
    public function getBalanceAttribute()
    {
        try {
            // Check if deposits table exists
            if (!Schema::hasTable('deposits')) {
                return 0;
            }

            // Get total approved deposits
            $totalDeposits = (float) ($this->deposits()
                ->where('status', 1) // approved
                ->sum('amount') ?? 0);

            // Check if orders table exists
            if (!Schema::hasTable('orders')) {
                return $totalDeposits;
            }

            // Get total wins (to add)
            $totalWins = (float) ($this->orders()
                ->where('type', 'win')
                ->where('type_money', 'balance_withdrawal')
                ->where('status', 1)
                ->sum('amount') ?? 0);

            // Get total losses (already deducted)
            $totalLosses = (float) ($this->orders()
                ->where('type', 'loss')
                ->where('type_money', 'balance_withdrawal')
                ->where('status', 1)
                ->sum('amount') ?? 0);

            // Balance = Deposits + Wins - Losses
            return max(0, $totalDeposits + $totalWins - $totalLosses);
        } catch (\Exception $e) {
            \Log::error('Balance calculation error for user ' . $this->id . ': ' . $e->getMessage());
            return 0;
        }
    }

    /**
     * Get formatted balance for display.
     */
    public function getFormattedBalanceAttribute()
    {
        return number_format($this->balance, 2, '.', ',');
    }
}
