<?php

namespace App\Http\Controllers;

use App\Models\GameRound;
use App\Models\Order;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Schema;

class HistoryController extends Controller
{
    public function index(Request $request)
    {
        try {
            $user = Auth::user();
            $filter = $request->get('filter', 'all');

            // Check if game_rounds table exists, if not, use orders table as fallback
            if (Schema::hasTable('game_rounds')) {
                return $this->getHistoryFromGameRounds($user, $filter);
            } elseif (Schema::hasTable('orders')) {
                return $this->getHistoryFromOrders($user, $filter);
            } else {
                return response()->json([
                    'success' => true,
                    'history' => [],
                    'stats' => [
                        'total_games' => 0,
                        'total_won' => 0,
                        'total_lost' => 0,
                    ],
                ]);
            }
        } catch (\Exception $e) {
            \Log::error('History API error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to load history',
                'history' => [],
                'stats' => [
                    'total_games' => 0,
                    'total_won' => 0,
                    'total_lost' => 0,
                ],
                'error' => config('app.debug') ? $e->getMessage() : null,
            ], 500);
        }
    }

    /**
     * Get history from game_rounds table (preferred method)
     */
    private function getHistoryFromGameRounds($user, $filter)
    {
        $query = GameRound::where('user_id', $user->id)
            ->where('result', '!=', 'no_bet'); // Only rounds where user placed a bet

        // Apply filters
        if ($filter === 'win') {
            $query->where('result', 'win');
        } elseif ($filter === 'loss') {
            $query->where('result', 'loss');
        } elseif ($filter === 'today') {
            $query->whereDate('created_at', today());
        } elseif ($filter === 'week') {
            $query->whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()]);
        } elseif ($filter === 'month') {
            $query->whereMonth('created_at', now()->month)
                  ->whereYear('created_at', now()->year);
        }

        $rounds = $query->orderBy('created_at', 'desc')->limit(100)->get();

        // Format history for display
        $history = $rounds->map(function ($round) {
            return [
                'created_at' => $round->created_at->toDateTimeString(),
                'bet_amount' => floatval($round->bet_amount),
                'multiplier' => floatval($round->multiplier),
                'win_amount' => floatval($round->win_amount),
                'type' => $round->result,
            ];
        })->toArray();

        // Calculate stats
        $totalGames = GameRound::where('user_id', $user->id)
            ->where('result', '!=', 'no_bet')
            ->when($filter === 'today', fn($q) => $q->whereDate('created_at', today()))
            ->when($filter === 'week', fn($q) => $q->whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()]))
            ->when($filter === 'month', fn($q) => $q->whereMonth('created_at', now()->month)->whereYear('created_at', now()->year))
            ->count();

        $totalWon = GameRound::where('user_id', $user->id)
            ->where('result', 'win')
            ->when($filter === 'today', fn($q) => $q->whereDate('created_at', today()))
            ->when($filter === 'week', fn($q) => $q->whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()]))
            ->when($filter === 'month', fn($q) => $q->whereMonth('created_at', now()->month)->whereYear('created_at', now()->year))
            ->when($filter === 'win', fn($q) => $q)
            ->when($filter === 'loss', fn($q) => $q->whereRaw('1=0')) // Empty query for loss filter
            ->sum('win_amount') ?? 0;

        $totalLost = GameRound::where('user_id', $user->id)
            ->whereIn('result', ['win', 'loss'])
            ->when($filter === 'today', fn($q) => $q->whereDate('created_at', today()))
            ->when($filter === 'week', fn($q) => $q->whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()]))
            ->when($filter === 'month', fn($q) => $q->whereMonth('created_at', now()->month)->whereYear('created_at', now()->year))
            ->when($filter === 'win', fn($q) => $q->whereRaw('1=0')) // Empty query for win filter
            ->when($filter === 'loss', fn($q) => $q->where('result', 'loss'))
            ->sum('bet_amount') ?? 0;

        return response()->json([
            'success' => true,
            'history' => $history,
            'stats' => [
                'total_games' => $totalGames,
                'total_won' => $totalWon,
                'total_lost' => $totalLost,
            ],
        ]);
    }

    /**
     * Fallback method using orders table
     */
    private function getHistoryFromOrders($user, $filter)
    {
        $query = Order::where('user_id', $user->id)
            ->where('game', 'aviator')
            ->where('type_money', 'balance_withdrawal');

        // Apply filters
        if ($filter === 'win') {
            $query->where('type', 'win');
        } elseif ($filter === 'loss') {
            $query->where('type', 'loss');
        } elseif ($filter === 'today') {
            $query->whereDate('created_at', today());
        } elseif ($filter === 'week') {
            $query->whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()]);
        } elseif ($filter === 'month') {
            $query->whereMonth('created_at', now()->month)
                  ->whereYear('created_at', now()->year);
        }

        $allOrders = $query->orderBy('created_at', 'desc')->limit(200)->get();

        // Group orders by transaction_id or game_uuid
        $groupedOrders = [];
        foreach ($allOrders as $order) {
            $gameId = $order->transaction_id ?: $order->game_uuid ?: uniqid();
            if (!isset($groupedOrders[$gameId])) {
                $groupedOrders[$gameId] = [];
            }
            $groupedOrders[$gameId][] = $order;
        }

        // Format history for display
        $history = [];
        foreach ($groupedOrders as $gameId => $orderGroup) {
            $betOrder = collect($orderGroup)->firstWhere('type', 'loss');
            $winOrder = collect($orderGroup)->firstWhere('type', 'win');
            
            if ($betOrder) {
                $multiplier = $winOrder && $betOrder->amount > 0 
                    ? round(floatval($winOrder->amount) / floatval($betOrder->amount), 2) 
                    : 0;
                
                $history[] = [
                    'created_at' => $betOrder->created_at->toDateTimeString(),
                    'bet_amount' => floatval($betOrder->amount),
                    'multiplier' => $multiplier,
                    'win_amount' => $winOrder ? floatval($winOrder->amount) : 0,
                    'type' => $winOrder ? 'win' : 'loss',
                ];
            }
        }
        
        // Sort by date descending and limit
        usort($history, function($a, $b) {
            return strtotime($b['created_at']) - strtotime($a['created_at']);
        });
        
        $history = array_slice($history, 0, 100);

        // Calculate stats
        $totalGames = count($history);
        $totalWon = collect($history)->where('type', 'win')->sum('win_amount');
        $totalLost = collect($history)->sum('bet_amount');
        
        if ($filter === 'win') {
            $totalLost = 0;
        } elseif ($filter === 'loss') {
            $totalWon = 0;
        }

        return response()->json([
            'success' => true,
            'history' => $history,
            'stats' => [
                'total_games' => $totalGames,
                'total_won' => $totalWon,
                'total_lost' => $totalLost,
            ],
        ]);
    }
}
