<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Order;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class BalanceController extends Controller
{
    /**
     * Get the authenticated user's balance.
     */
    public function getBalance()
    {
        try {
            $user = Auth::user();
            
            if (!$user) {
                return response()->json([
                    'error' => 'Unauthorized',
                    'balance' => 0,
                    'formatted_balance' => '0.00',
                ], 401);
            }
            
            // Calculate balance safely
            $balance = $user->balance ?? 0;
            $formattedBalance = $user->formatted_balance ?? '0.00';
            
            return response()->json([
                'balance' => $balance,
                'formatted_balance' => $formattedBalance,
            ]);
        } catch (\Exception $e) {
            \Log::error('Balance API Error: ' . $e->getMessage());
            return response()->json([
                'error' => 'Failed to fetch balance',
                'balance' => 0,
                'formatted_balance' => '0.00',
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Deduct balance for a bet.
     */
    public function deductBalance(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'game_uuid' => 'required|string',
        ]);

        $user = Auth::user();
        $amount = $request->amount;

        if ($user->balance < $amount) {
            return response()->json([
                'success' => false,
                'message' => 'Insufficient balance!',
            ], 400);
        }

        // Create order for bet (loss/deduction)
        $order = Order::create([
            'user_id' => $user->id,
            'transaction_id' => uniqid('bet_'),
            'game' => 'aviator',
            'game_uuid' => $request->game_uuid,
            'type' => 'loss',
            'type_money' => 'balance_withdrawal',
            'amount' => $amount,
            'providers' => 'aviator',
            'refunded' => 0,
            'status' => 1,
            'hash' => md5(uniqid()),
        ]);

        return response()->json([
            'success' => true,
            'balance' => $user->fresh()->balance,
            'formatted_balance' => $user->fresh()->formatted_balance,
            'order_id' => $order->id,
        ]);
    }

    /**
     * Add balance for a win (cashout).
     */
    public function addBalance(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'game_uuid' => 'required|string',
        ]);

        $user = Auth::user();

        // Create order for win
        $order = Order::create([
            'user_id' => $user->id,
            'transaction_id' => uniqid('win_'),
            'game' => 'aviator',
            'game_uuid' => $request->game_uuid,
            'type' => 'win',
            'type_money' => 'balance_withdrawal',
            'amount' => $request->amount,
            'providers' => 'aviator',
            'refunded' => 0,
            'status' => 1,
            'hash' => md5(uniqid()),
        ]);

        return response()->json([
            'success' => true,
            'balance' => $user->fresh()->balance,
            'formatted_balance' => $user->fresh()->formatted_balance,
            'order_id' => $order->id,
        ]);
    }
}
